(ns works-hub.pachyderm.products
  (:refer-clojure :exclude [list update])
  (:require [clojure.string :as str]
            [works-hub.pachyderm.transforms :as trs]
            [works-hub.pachyderm.util :refer (map->request-options map->str-ks)])
  (:import [clojure.lang IPersistentMap]
           [com.stripe.exception InvalidRequestException]
           [com.stripe.model Product]))

(defn list
  ([]
   (list {}))
  ([^IPersistentMap attrs]
   (list attrs nil))
  ([^IPersistentMap attrs ^IPersistentMap ro-opts]
   (trs/products-coll->seq (Product/list
                            (map->str-ks attrs)
                            (map->request-options ro-opts)))))

(defn create
  ([^IPersistentMap attrs]
   (create attrs nil))
  ([^IPersistentMap attrs ^IPersistentMap ro-opts]
   (trs/product->map (Product/create
                   (map->str-ks attrs)
                   (map->request-options ro-opts)))))

(defn retrieve
  ([^String id]
   (retrieve id nil))
  ([^String id ^IPersistentMap ro-opts]
   (try
     (trs/product->map (Product/retrieve
                        id
                        (map->request-options ro-opts)))
     (catch InvalidRequestException e
       (when-not (str/starts-with? (.getMessage e) "No such product")
         (throw e))))))

(defn update
  ([^IPersistentMap product ^IPersistentMap attrs]
   (update product attrs nil))
  ([^IPersistentMap product ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ product)]
     (trs/product->map (.update o
                                (map->str-ks attrs)
                                (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "products/update only accepts maps returned by products/create, products/retrieve, and products/list")))))

(defn delete
  ([^IPersistentMap product]
   (delete product nil))
  ([^IPersistentMap product ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ product)]
     (trs/product->map (.delete o (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "products/delete only accepts maps returned by products/create, products/retrieve, and products/list")))))
