(ns works-hub.pachyderm.invoices
  (:refer-clojure :exclude [list update])
  (:require [works-hub.pachyderm.transforms :as trs]
            [works-hub.pachyderm.util :refer (map->request-options map->str-ks)])
  (:import [clojure.lang IPersistentMap]
           [com.stripe.model Invoice]))

(defn create
  ([^IPersistentMap customer]
   (create customer {}))
  ([^IPersistentMap customer ^IPersistentMap attrs]
   (create customer attrs nil))
  ([^IPersistentMap customer ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if (:__origin__ customer)
     (trs/invoice->map (Invoice/create
                        (map->str-ks (merge attrs {:customer (:id customer)}))
                        (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "invoices/create only accepts maps returned by fns in the customers namespace")))))

(defn retrieve
  ([^String id]
   (retrieve id nil))
  ([^String id ^IPersistentMap ro-opts]
   (trs/invoice->map (Invoice/retrieve id
                                       (map->request-options ro-opts)))))

(defn update
  ([^IPersistentMap invoice ^IPersistentMap attrs]
   (update invoice attrs nil))
  ([^IPersistentMap invoice ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ invoice)]
     (trs/invoice->map (.update o
                                (map->str-ks attrs)
                                (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "invoice/update only accepts maps returned by invoice/create and invoice/retrieve")))))

(defn upcoming
  ([^IPersistentMap customer]
   (upcoming customer {}))
  ([^IPersistentMap customer ^IPersistentMap attrs]
   (upcoming customer attrs nil))
  ([^IPersistentMap customer ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if (:__origin__ customer)
     (trs/invoice->map (Invoice/upcoming (merge {"customer" (:id customer)}
                                                (map->str-ks attrs))
                                         (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "invoice/upcoming only accepts customers returned by customers/create, customers/retrieve, and customers/list")))))

(defn prorated?
  "True if at least one item and all items are prorated."
  [^IPersistentMap invoice]
  (let [items (:invoice-items invoice)]
    (and (-> items count pos?)
         (every? :proration items))))

(defn list
  ([]
   (list {}))
  ([^IPersistentMap params]
   (list params nil))
  ([^IPersistentMap params ^IPersistentMap ro-opts]
   (trs/invoice-coll->seq
    (Invoice/list (map->str-ks params)
                  (map->request-options ro-opts)))))

(defn pay
  ([^IPersistentMap invoice]
   (pay invoice nil))
  ([^IPersistentMap invoice ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ invoice)]
     (trs/invoice->map (.pay o
                             (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "invoice/update only accepts maps returned by invoice/create and invoice/retrieve")))))
