(ns works-hub.pachyderm.subscriptions
  (:refer-clojure :exclude [list update])
  (:require [works-hub.pachyderm.transforms :as trs]
            [works-hub.pachyderm.util :refer (map->request-options map->str-ks)])
  (:import [clojure.lang IPersistentMap IPersistentStack]
           [com.stripe.model Subscription]))

(defn list
  ([]
   (list {}))
  ([^IPersistentMap attrs]
   (list attrs nil))
  ([^IPersistentMap attrs ^IPersistentMap ro-opts]
   (trs/subscriptions-coll->seq (Subscription/list
                                 (map->str-ks attrs)
                                 (map->request-options ro-opts)))))

(defn retrieve
  ([^String id]
   (retrieve id {}))
  ([^String id ^IPersistentMap attrs]
   (retrieve id attrs nil))
  ([^String id ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (let [cs (list attrs ro-opts)]
     (some #(when (= (:id %) id) %) cs))))

(defn create
  "Creates a subscription.
   - customer: A customer map returned from an API fn in the customer ns.
   - plans:    A collection of `{:plan \"<plan-id>}\" :quantity <quantity> }`
   - attrs:    Attributes (https://stripe.com/docs/api?lang=java#create_subscription)
   - ro-opts:  RequestOption options (https://github.com/Works-Hub/pachyderm/blob/master/src/works_hub/pachyderm/util.clj#L8)"
  ([^IPersistentMap customer ^IPersistentStack plans]
   (create customer plans {}))
  ([^IPersistentMap customer ^IPersistentStack plans ^IPersistentMap attrs]
   (create customer plans attrs nil))
  ([^IPersistentMap customer ^IPersistentStack plans ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if (:__origin__ customer)
     (trs/subscription->map (Subscription/create
                             (map->str-ks (merge attrs {:customer (:id customer)
                                                        :items (map map->str-ks plans)}))
                             (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "subscriptions/create only accepts maps returned by fns in the customers namespace")))))

(defn update
  ([^IPersistentMap subscription ^IPersistentMap attrs]
   (update subscription attrs nil))
  ([^IPersistentMap subscription ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ subscription)]
     (trs/subscription->map (.update o
                                     (map->str-ks attrs)
                                     (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "subscriptions/update only accepts subscriptions returned by fns in the customers or subscriptions namespaces")))))

(defn cancel
  ([^IPersistentMap subscription]
   (cancel subscription {}))
  ([^IPersistentMap subscription ^IPersistentMap attrs]
   (cancel subscription attrs nil))
  ([^IPersistentMap subscription ^IPersistentMap attrs ^IPersistentMap ro-opts]
   (if-let [o (:__origin__ subscription)]
     (trs/subscription->map (.cancel o
                                     (map->str-ks attrs)
                                     (map->request-options ro-opts)))
     (throw (IllegalArgumentException.
             "subscriptions/cancel only accepts maps returned by customers/subscribe and subscriptions/retrieve")))))
