(ns witan.models.dem.ccm.fert.fertility
  (:require [witan.models.dem.ccm.components-functions :as cf]
            [witan.workspace-api :refer [defworkflowfn]]
            [clojure.core.matrix.dataset :as ds]
            [witan.datasets :as wds]
            [witan.models.dem.ccm.schemas :refer :all]
            [schema.core :as s]
            [witan.workspace-api.utils :as utils]
            [witan.models.dem.ccm.models-utils :as m-utils]))

(defn create-popn-at-risk-birth
  "Creates a population at risk for calculating historic age specific
  fertility rates. Takes in the historic population and a base year."
  [historic-population base-year]
  (-> historic-population
      (wds/select-from-ds {:year (dec base-year)})
      (ds/rename-columns {:popn :popn-at-risk})))

(defn calc-asfr-birth-data-year
  "Takes the population at risk for the birth data year and ONS
  projections for births by age of mother. Returns fertility
  rates for the birth data year."
  [popn-at-risk-birth-data-year hist-births-by-age-mother]
  (-> hist-births-by-age-mother
      (wds/join popn-at-risk-birth-data-year [:gss-code :sex :age])
      (wds/add-derived-column :births [:births] (fn [b] (or b 0.0)))
      (wds/add-derived-column :fert-rate-birth-data-year [:births :popn-at-risk]
                              (fn [b p] (wds/safe-divide [b p])))
      (ds/select-columns [:gss-code :sex :age :fert-rate-birth-data-year])))

(defn calc-estimated-births
  "Takes in the population at risk for fert-base-year and the dataset
  with asfr for the base year of the birth data, to calculate the
  estimated births which is totalled by gss-code and year."
  [popn-at-risk-fert-base-year asfr-birth-data-year]
  (-> asfr-birth-data-year
      (wds/join popn-at-risk-fert-base-year [:gss-code :sex :age])
      (wds/add-derived-column :births [:popn-at-risk :fert-rate-birth-data-year]
                              (fn [p b] (* ^double p b)))
      (wds/add-derived-column :year [:year] inc)
      (wds/rollup :sum :births [:gss-code :year])
      (ds/rename-columns {:births :estimated-births})))

(defn calc-actual-births
  "Takes in the historic data of total births, filters for
  the base year and totals the number of births for that year
  and gss-code."
  [historic-births fert-base-year]
  (-> historic-births
      (wds/select-from-ds {:year fert-base-year})
      (wds/rollup :sum :births [:gss-code :year])
      (ds/rename-columns {:births :actual-births})))

(defn calc-scaling-factors
  "Takes in the estimated and actual births. Divides them to get
  the scaling factors for each year and gss-code."
  [actual-births estimated-births]
  (-> estimated-births
      (wds/join actual-births [:gss-code :year])
      (wds/add-derived-column :scaling-factor [:actual-births
                                               :estimated-births]
                              (fn [a e] (wds/safe-divide [a e])))
      (ds/select-columns [:gss-code :year :scaling-factor])))

(defn calc-scaled-fert-rates
  "Takes in scaling factor and base fertility rates to
  return the scaled fertility rates."
  [scaling-factors asfr-birth-data-year]
  (-> scaling-factors
      (wds/join asfr-birth-data-year [:gss-code])
      (wds/add-derived-column :fert-rate [:scaling-factor
                                          :fert-rate-birth-data-year]
                              (fn [s b] (* ^double s b)))))

(defworkflowfn calculate-historic-asfr
  "Takes in three datasets: the historic total births, the historic
   population, the base age-specific fertility rate and the base year.
   Returns a dataset containing the historic age specific fertility rates."
  {:witan/name :ccm-fert/calc-hist-asfr
   :witan/version "1.0.0"
   :witan/input-schema {:historic-births-by-age-mother BirthsAgeSexMotherSchema
                        :historic-population PopulationSchema
                        :historic-births BirthsSchema}
   :witan/param-schema {:fert-base-year (s/constrained s/Int m-utils/year?)}
   :witan/output-schema {:historic-asfr HistASFRSchema}
   :witan/exported? true}
  [{:keys [base-asfr historic-births-by-age-mother historic-population historic-births]}
   {:keys [fert-base-year]}]
  (let [birth-data-year (reduce max (ds/column
                                     historic-births-by-age-mother :year))
        _ (utils/property-holds? birth-data-year m-utils/year? (str birth-data-year " is not a year"))
        popn-at-risk-birth-data-year (-> historic-population
                                         (create-popn-at-risk-birth birth-data-year)
                                         (ds/select-columns
                                          [:gss-code :sex :age :popn-at-risk]))
        asfr-birth-data-year (calc-asfr-birth-data-year popn-at-risk-birth-data-year
                                                        historic-births-by-age-mother)
        popn-at-risk-fert-proj-base-year (create-popn-at-risk-birth historic-population fert-base-year)
        estimated-births (calc-estimated-births popn-at-risk-fert-proj-base-year asfr-birth-data-year)]
    {:historic-asfr
     (-> historic-births
         (calc-actual-births fert-base-year)
         (calc-scaling-factors estimated-births)
         (calc-scaled-fert-rates asfr-birth-data-year)
         (ds/select-columns [:gss-code :sex :age :year :fert-rate]))}))

(defworkflowfn project-asfr-finalyearhist-fixed
  "Takes dataset of historic age specific fertility rates, and parameter
   for the base year of fertility data. Returns dataset with projected
   age specific fertility rates, calculated using the jumpoff year average
   method (see docs)."
  {:witan/name :ccm-fert/project-asfr-finalyearhist-fixed
   :witan/version "1.0.0"
   :witan/input-schema {:historic-asfr HistASFRSchema}
   :witan/output-schema {:initial-projected-fertility-rates ProjFixedASFRSchema}
   :witan/exported? true}
  [{:keys [historic-asfr]} _]
  (let [final-year (reduce max (ds/column historic-asfr :year))
        _ (utils/property-holds? final-year m-utils/year? (str final-year " is not a year"))
        final-year-hist (wds/select-from-ds historic-asfr {:year final-year})]
    {:initial-projected-fertility-rates
     (ds/select-columns final-year-hist [:gss-code :sex :age :fert-rate])}))

(defworkflowfn project-births-from-fixed-rates
  "Takes a dataset with population at risk from the current year of the projection
  loop and another dataset with fixed fertility rates for the population. Returns a
  dataset with a column of births, which are the product of popn at risk & the rates"
  {:witan/name :ccm-fert/project-births-fixed-rates
   :witan/version "1.0.0"
   :witan/input-schema {:initial-projected-fertility-rates ProjFixedASFRSchema
                        :population-at-risk PopulationAtRiskSchema}
   :witan/output-schema {:births-by-age-sex-mother BirthsAgeSexMotherSchema}
   :witan/exported? true}
  [{:keys [initial-projected-fertility-rates population-at-risk]} _]
  (let [projected-births (cf/project-component-fixed-rates
                          population-at-risk
                          initial-projected-fertility-rates
                          :fert-rate :births)]
    {:births-by-age-sex-mother projected-births}))

(defn- gather-births-by-sex
  "Given a dataset with columns :gss-code, :m, and :f (where :m and :f are
   male and female births), gathers births data into :births column and
   sex into :sex column, returning a new dataset. Standing for a universal
   gather function in witan.datasets similar to gather in tidyR"
  [births-by-mf]
  (let [births-f (-> births-by-mf
                     (ds/add-column :sex ["F"])
                     (ds/rename-columns {:f :births})
                     (ds/select-columns [:gss-code :sex :births]))
        births-m (-> births-by-mf
                     (ds/add-column :sex ["M"])
                     (ds/rename-columns {:m :births})
                     (ds/select-columns [:gss-code :sex :births]))]
    (ds/join-rows births-m births-f)))

(defworkflowfn combine-into-births-by-sex
  "Takes dataset of historic age specific fertility rates, and parameter
   for the base year of fertility data. Returns dataset with projected
   age specific fertility rates, calculated using the jumpoff year average
   method (see docs)."
  {:witan/name :ccm-fert/combine-into-births-by-sex
   :witan/version "1.0.0"
   :witan/input-schema {:births-by-age-sex-mother BirthsAgeSexMotherSchema}
   :witan/param-schema {:proportion-male-newborns double}
   :witan/output-schema {:births BirthsBySexSchema}
   :witan/exported? true}
  [{:keys [births-by-age-sex-mother]} {:keys [proportion-male-newborns]}]
  (let [births-by-sex (-> births-by-age-sex-mother
                          (wds/rollup :sum :births [:gss-code])
                          (wds/add-derived-column :m [:births]
                                                  (fn [b] (double (* proportion-male-newborns b))))
                          (wds/add-derived-column :f [:births]
                                                  (fn [b] (double
                                                           (* (- 1 proportion-male-newborns) b))))
                          (ds/select-columns [:gss-code :m :f])
                          gather-births-by-sex)]
    {:births births-by-sex}))
