(ns wiseloong.utils
  (:require [clj-time.core :as c]
            [clj-time.format :as f]
            [buddy.core.hash :as hash]
            [buddy.core.codecs :as codecs]
            [cheshire.core :as cjson]
            [clojure.string :as str])
  (:import (org.joda.time DateTime)))

(defn DateTime? [d]
  (instance? DateTime d))

(def PRC
  (c/time-zone-for-id "PRC"))

(defn date-format [k]
  (k {:date-time "yyyy-MM-dd HH:mm:ss"
      :date      "yyyy-MM-dd"
      :date-ym   "yyyyMM"
      :file-tail "_yyyyMMddHHmmss"}))

(defn unparse
  ([dt] (unparse dt :date))
  ([dt format]
   (if (DateTime? dt)
     (let [fmt (f/formatter (date-format format) PRC)]
       (f/unparse fmt dt))
     dt)))

(defn now-string "当前时间-字符串"
  ([] (now-string :date-time))
  ([format]
   (unparse (c/now) format)))

(defn now []
  (c/to-time-zone (c/now) PRC))

(defn time-string
  ([date] (time-string date :date))
  ([date format]
   (letfn [(l [[k v]] [k (unparse v format)])]
     (->> (map l date)
          (into {})))))

(defn map-time-string
  ([m] (map time-string m))
  ([m format]
   (map #(time-string % format) m)))

(defn for-map-val [m f]
  (letfn [(l [[k v]] [k (f v)])]
    (->> (map l m)
         (into {}))))

(defn string-time
  ([date key] (string-time date key :date))
  ([date key format]
   (let [fmt (f/formatter (date-format format) PRC)]
     (assoc date key (f/parse fmt (key date))))))

(defn page-size [d]
  (let [page (:page d)
        size (:size d)]
    (if (and page size)
      (let [start (* size (- page 1))]
        (assoc d :start start))
      d)))

(defn insert-time [d]
  (let [now (c/now)]
    (merge d {:create_date now :modify_date now :is_valid 1})))

(defn update-time [d]
  (merge d {:modify_date (c/now)}))

(defn parse-int [s]
  (if s
    (Integer/parseInt (re-find #"-?\d+" s))
    0))

(defn md5 [s]
  (-> s hash/md5 codecs/bytes->hex))

(defn diff-data "以m1为基准，获取2哥map的不同值，组成{:fields key, :new v1, :old v2}"
  [m1 m2]
  (letfn [(merge-data [[k v]] {:fields k :new v :old (get m2 k)})
          (diff [{:keys [new old]}] (not (= new old)))]
    (->> m1
         (map merge-data)
         (filter diff))))

(defn collection-map-key "把map里按照k的值合并"
  ([m k] (collection-map-key m k {}))
  ([m k r]
   (let [one (first m)]
     (if-not (nil? one)
       (let [k1 (keyword (k one))
             v1 (conj (k1 r) one)]
         (collection-map-key (next m) k (assoc r k1 v1)))
       r))))

(defn any-empty? "判断是否为空，遇到数字类型不报错" [a]
  (or (nil? a) (and (string? a) (str/blank? a))))

(defn join-comma "去除空元素，把集合合并成字符串,用','分割"
  [coll] (let [coll (remove any-empty? coll)]
           (str/join "," coll)))

(defn parse-string "字符串转json，转成keyword形式 {:id id}"
  [str] (cjson/parse-string str true))
