(ns simplelog.core
  (:require 
    [clj-statsd :as stats]
    [clj-stacktrace.repl :as stack])
  (:import 
    [java.util.Date]
    [java.text.DateFormat]))

(defn- timestamp []
  (str "[" (.toGMTString (java.util.Date.)) "]"))

(defn- thread-id []
  (.getId (Thread/currentThread)))

(defonce enabled-levels (atom #{}))

(defn define-level [level]
  (swap! enabled-levels conj level)
  `(defn ~level [& anything#]
     (when (@enabled-levels '~level)
       (apply println (timestamp)
         (str "[thread=" (thread-id) "]")
         (.toUpperCase (str "[" '~level "]"))
         anything#))))

(defmacro log-levels [& levels]
  `(do ~@(map define-level levels)))

(defmacro benchmark
  "Evaluates expr and prints the time it took.  Returns the value of
 expr."
  [name & body]
  `(let [start# (. System (nanoTime))
         ret# (do ~@body)
         time# (/ (double (- (. System (nanoTime)) start#)) 1000000.0)]
     (bench (str "[" ~name "] Elapsed time: " time# " msecs"))
     (stats/timing ~name (int time#))
     ret#))

(def print-exception stack/pst+)
(log-levels debug info warn error bench trace fatal spy)

(defn silence!
  [& levels]
  (doseq [level levels]
    (swap! enabled-levels disj (-> level name symbol))))

(defn enable!
  [& levels]
  (doseq [level levels]
    (swap! enabled-levels conj (-> level name symbol))))
