(ns whoneedszzz.orientdb-client-sql.client
  (:require [cheshire.core :as json]
            [clojure.spec.alpha :as s])
  (:import [com.orientechnologies.orient.core.db ODatabase
                                                 OrientDB
                                                 OrientDBConfig
                                                 OrientDBConfigBuilder
                                                 ODatabasePool
                                                 ODatabaseSession
                                                 ODatabaseType]
           [com.orientechnologies.orient.core.db.document ODatabaseDocumentRemote]
           [com.orientechnologies.orient.core.sql.executor OResult
                                                           OResultSet]
           [java.util Map]))

(defn- gen-o-config
  "Generates an OrientDB configuration map"
  [config]
  (-> (OrientDBConfigBuilder.)
      (.fromMap config)
      (.build)))

(defn- gen-o-map
  "Generates a map with string keys for OrientDB"
  [params]
  (into {}
        (for [[k v] params]
          [(name k) v])))

(defn- gen-result-map
  "Generates a map of the properties of a given OrientDB result"
  [^OResult result sort? keywords?]
  (let [json-str (.toJSON result)
        res-map (json/parse-string json-str keywords?)]
    (if sort?
      (into (sorted-map) res-map)
      res-map)))

(defn- resultset->maps
  "Converts an OResultSet to a vector of map(s)"
  [^OResultSet result-set sort? keywords?]
  (let [map-vec (atom [])]
    (while (.hasNext result-set)
      (let [result (.next result-set)
            result-map (gen-result-map result sort? keywords?)]
        (swap! map-vec conj result-map)))
    @map-vec))

(defn begin
  "Begins an OrientDB transaction"
  [^ODatabase session]
  (.begin session))

(defn command
  "Returns the result of the given session, command, and parameters as a vector of maps optionally sorted and using keywords"
  [{:keys [^ODatabase session ^String command params sort? keywords?]}]
  (let [o-params ^Map (gen-o-map params)
        result-set (.command session command o-params)
        results (resultset->maps result-set sort? keywords?)]
    (.close result-set)
    results))

(defn commit
  "Commits an OrientDB transaction"
  [^ODatabase session]
  (.commit session))

(defn connect
  "Connect to a remote OrientDB instance with required and optional properties
  Note: :root-user & :root-password may be nil, but are required for root tasks"
  [{:keys [url root-user root-password config db-name db-user db-password pool?]}]
  (let [db-config (gen-o-config config)
        db (OrientDB. url root-user root-password db-config)]
    (if pool?
      (let [pool (ODatabasePool. ^OrientDB db ^String db-name ^String db-user ^String db-password)]
        {:db db :session (.acquire pool)})
      (let [session (.open db db-name db-user db-password)]
        {:db db :session session}))))

(defn close-db
  "Close the database connection"
  [db]
  (.close db)
  (not (.isOpen db)))

(defn close-session
  "Close the database session"
  [session]
  (.close session)
  (.isClosed session))

(defn create-db
  "Creates an OrientDB database
  Note: Root credentials must have been provided in connect"
  [{:keys [^OrientDB db db-name db-type config]}]
  (let [db-config (gen-o-config config)]
    (case db-type
      "plocal" (.create db db-name ODatabaseType/PLOCAL db-config)
      "memory" (.create db db-name ODatabaseType/MEMORY db-config))))

(defn drop-db
  "Drops given OrientDB database
  Note: Root credentials must have been provided in connect"
  [{:keys [^OrientDB db db-name]}]
  (.drop db db-name))

(defn query
  "Returns the result of the given session, query, and parameters as a vector of maps optionally sorted and using keywords"
  [{:keys [^ODatabase session ^String query params sort? keywords?]}]
  (let [o-params ^Map (gen-o-map params)
        result-set (.query session query o-params)
        results (resultset->maps result-set sort? keywords?)]
    (.close result-set)
    results))

(defn rollback
  "Rolls back an OrientDB transaction"
  [^ODatabase session]
  (.rollback session))

(defn script
  "Executes a script"
  [{:keys [^ODatabaseSession session ^String lang ^String script params keywords?]}]
  (let [o-params ^Map (gen-o-map params)
        result-set (.execute session lang script o-params)
        results (resultset->maps result-set false keywords?)]
    (.close result-set)
    results))

; Specs
(s/def ::command string?)
(s/def ::config map?)
(s/def ::db #(instance? OrientDB %))
(s/def ::db-name string?)
(s/def ::db-password string?)
(s/def ::db-type (s/and string? #(re-matches #"plocal|memory" %)))
(s/def ::db-user string?)
(s/def ::keywords? boolean?)
(s/def ::query string?)
(s/def ::params map?)
(s/def ::pool? boolean?)
(s/def ::result #(instance? OResult %))
(s/def ::result-set #(instance? OResultSet %))
(s/def ::root-password (s/or :s string? :n nil?))
(s/def ::root-user (s/or :s string? :n nil?))
(s/def ::session #(instance? ODatabaseDocumentRemote %))
(s/def ::sort? boolean?)
(s/def ::sorted-map (s/and sorted? map?))
(s/def ::url (s/and string? #(re-matches #"(remote|embedded):((\w+(-\w+)*)|\.*)+(\/\w+(-\w+)*)*(\/(\w+(-\w+)*))*" %)))
(s/def ::vec-of-maps (s/coll-of map? :into []))
(s/fdef begin
        :args (s/cat :session ::session)
        :ret ::session)
(s/fdef command
        :args (s/cat :args (s/keys :req-un [::session ::command ::params ::sort? ::keywords?]))
        :ret ::vec-of-maps)
(s/fdef commit
        :args (s/cat :session ::session)
        :ret ::session)
(s/fdef connect
        :args (s/cat :args (s/keys :req-un [::url ::root-user ::root-password ::config ::db-name ::db-user ::db-password ::pool?]))
        :ret ::session)
(s/fdef close-db
        :args (s/cat :db ::db)
        :ret boolean?)
(s/fdef close-session
        :args (s/cat :session ::session)
        :ret boolean?)
(s/fdef create-db
        :args (s/cat :args (s/keys :req-un [::db ::db-name ::db-type ::config]))
        :ret nil?)
(s/fdef drop-db
        :args (s/cat :args (s/keys :req-un [::db ::db-name])))
(s/fdef gen-o-config
        :args (s/cat :config ::config)
        :ret #(instance? OrientDBConfig %))
(s/fdef gen-o-map
        :args (s/cat :params ::params)
        :ret ::params)
(s/fdef gen-result-map
        :args (s/cat :result ::result :sort? ::sort? :keywords? ::keywords?)
        :ret ::sorted-map)
(s/fdef query
        :args (s/cat :args (s/keys :req-un [::session ::query ::params ::sort? ::keywords?]))
        :ret ::vec-of-maps)
(s/fdef resultset->maps
        :args (s/cat :result-set ::result-set :sort? ::sort? :keywords? ::keywords?)
        :ret ::vec-of-maps)
(s/fdef rollback
        :args (s/cat :session ::session)
        :ret ::session)
(s/fdef script
        :args (s/cat :args (s/keys :req-un [::session ::lang ::script ::options ::sort? ::keywords?]))
        :ret ::vec-of-maps)
