(ns om-image-grid.core
  (:require [om.core :as om :include-macros true]
            [om.dom :as dom :include-macros true]
            [goog.events :as events]
            [goog.events.EventType]))

(defn grid-sizes [widths cont-width max-height spacing]
  ((fn step [v row width coll]
     (if-let [xs (seq coll)]
       (let [x        (first xs)
             more     (rest xs)
             width'   (+ width x spacing)
             row'     (conj row x)
             row-w    (apply + row')]
         (if (>= width' cont-width)
           (let [row-factor (/ row-w (- cont-width (* (count row') spacing)))
                 row-height (/ max-height row-factor)
                 row-widths (map #(/ % row-factor) row')
                 row-sizes  (reduce #(conj %1 [%2 row-height]) [] row-widths)
                 difference (-> (- (apply + row-widths)
                                   (->> (map js/Math.floor row-widths)
                                        (apply +)))
                                js/Math.floor)
                 row-sizes' (map (fn [[w h] p]
                                   [(-> w js/Math.floor (+ p)) (js/Math.floor h)])
                                 row-sizes
                                 (concat (repeat difference 1) (repeat 0)))]
             (step (concat v row-sizes') [] 0 more))
           (step v row' width' more)))
       (vec v)))
   [] [] 0 widths))

(defn get-widths
  [images max-height]
  (map #(/ (:width %) (/ (:height %) max-height)) images))

(defn image-grid
  [images owner {:keys [compo spacing max-height]}]
  (reify
    om/IDidMount
    (did-mount [_]
      (let [node (om/get-node owner "container")]
        (om/set-state! owner :cont-width (.-clientWidth node))
        (events/listen js/window goog.events.EventType.RESIZE
          (fn [event]
            (let [new-width (.-clientWidth node)]
              (when (not= (om/get-state owner :cont-width) new-width)
                (om/set-state! owner :cont-width new-width)))))))
    om/IRenderState
    (render-state [_ {:keys [cont-width]}]
      (apply dom/div #js {:ref "container"}
        (let [sizes (when cont-width
                      (-> (get-widths images max-height)
                          (grid-sizes (- cont-width 1) max-height spacing)))]
          (map (fn [u i]
                 (let [[width height] (get sizes i)]
                   (om/build compo u {:state {:grid-width width
                                              :grid-height height}
                                      :opts {:max-height max-height}})))
                   images (range)))))))
