(ns om-image-grid.cards
  (:require-macros [devcards.core :refer [defcard is are= are-not=]])
  (:require [devcards.core :as dc :include-macros true]
            [om.core :as om :include-macros true]
            [om.dom :as dom :include-macros true]
            [om-image-grid.core :refer [image-grid]]))

(enable-console-print!)

(devcards.core/start-devcard-ui!)
(devcards.core/start-figwheel-reloader!)

(defcard devcard-intro
  (dc/markdown-card
   "# Om-image-grid

    Github: [https://github.com/whodidthis/om-image-grid](https://github.com/whodidthis/om-image-grid).

    An om component for creating justified image grids.
    ### Usage
    Takes images as a vector of maps
    that each have `:file` for path, and `:width` and `:height` for dimensions.
    ```
    [{:file \"image1.jpg\" :width 123 :height 456}
     {:file \"image2.jpg\" :width 246 :height 421}
     ...]
    ```
    Found from clojars with `[whodidthis/om-image-grid \"0.3.0\"]`.
    ```
    (:require [om-image-grid.core :refer [image-grid]])
    ```
    Building the grid requires an image `:compo` to use for each image,
    `:max-height` for images to display and additional `:spacing` each image
    needs horizontally.
    "))

(defcard desc-images-200-px
  (dc/markdown-card
    "### Example
     Define an image component and build it with `om-image-grid.core/image-grid`
     with max-height of the images to be shown and whatever additional horizontal
     spacing each image needs. Each images local state will be updated with
     `grid-width` and `grid-height`. Here each image has a padding of 2px so
     additional horizontal spacing required by each image is 4px.
     ```
     (defn image-200-compo [{:keys [file width height]} owner]
       (reify
         om/IRenderState
         (render-state [_ {:keys [grid-width grid-height]}]
           (dom/div #js {:style #js {:display \"inline-block\" :padding 2}}
             (dom/img #js {:src file :width (or grid-width width) :height (or grid-height height)})))))

     (om/build image-grid images {:opts {:compo image-200-compo
                                         :max-height 200
                                         :spacing 4}})
    ```"))

(def images-200
  [{:file "200/01.jpg" :width 140 :height 200} {:file "200/02.jpg" :width 133 :height 200}
   {:file "200/03.jpg" :width 132 :height 200} {:file "200/04.jpg" :width 406 :height 200}
   {:file "200/05.jpg" :width 150 :height 200} {:file "200/06.jpg" :width 151 :height 200}
   {:file "200/07.jpg" :width 136 :height 200} {:file "200/08.jpg" :width 160 :height 200}
   {:file "200/09.jpg" :width 140 :height 200} {:file "200/10.jpg" :width 267 :height 200}
   {:file "200/11.jpg" :width 300 :height 200} {:file "200/12.jpg" :width 155 :height 200}
   {:file "200/13.jpg" :width 154 :height 200} {:file "200/14.jpg" :width 135 :height 200}
   {:file "200/15.jpg" :width 140 :height 200} {:file "200/16.jpg" :width 148 :height 200}
   {:file "200/17.jpg" :width 300 :height 200} {:file "200/18.jpg" :width 147 :height 200}
   {:file "200/19.jpg" :width 158 :height 200} {:file "200/20.jpg" :width 126 :height 200}
   ])

(defn image-200-compo [{:keys [file width height]} owner]
  (reify
    om/IRenderState
    (render-state [_ {:keys [grid-width grid-height]}]
      (dom/div #js {:style #js {:display "inline-block" :padding 2}}
        (dom/img #js {:src file :width (or grid-width width) :height (or grid-height height)})))))

(defcard images-200-px
  (dc/om-root-card
    (fn [images owner]
      (om/component
        (dom/div #js {:style #js {:background-color "#428bca" :padding 2}}
          (om/build image-grid images {:opts {:compo image-200-compo
                                              :max-height 200
                                              :spacing 4}}))))
    (->> images-200 shuffle)))

(def images-300
  [{:file "300/01.jpg" :width 210 :height 300} {:file "300/02.jpg" :width 200 :height 300}
   {:file "300/03.jpg" :width 198 :height 300} {:file "300/04.jpg" :width 609 :height 300}
   {:file "300/05.jpg" :width 225 :height 300} {:file "300/06.jpg" :width 227 :height 300}
   {:file "300/07.jpg" :width 204 :height 300} {:file "300/08.jpg" :width 240 :height 300}
   {:file "300/09.jpg" :width 209 :height 300} {:file "300/10.jpg" :width 400 :height 300}
   {:file "300/11.jpg" :width 450 :height 300} {:file "300/12.jpg" :width 232 :height 300}
   {:file "300/13.jpg" :width 232 :height 300} {:file "300/14.jpg" :width 203 :height 300}
   {:file "300/15.jpg" :width 211 :height 300} {:file "300/16.jpg" :width 222 :height 300}
   {:file "300/17.jpg" :width 450 :height 300} {:file "300/18.jpg" :width 221 :height 300}
   {:file "300/19.jpg" :width 238 :height 300} {:file "300/20.jpg" :width 189 :height 300}
   ])

(defn image-300-compo [{:keys [file width height]} owner]
  (reify
    om/IRenderState
    (render-state [_ {:keys [grid-width grid-height]}]
      (dom/div #js {:style #js {:display "inline-block" :padding 5}}
        (dom/img #js {:src file :width (or grid-width width) :height (or grid-height height)})))))

(defcard images-300-px
  (dc/om-root-card
    (fn [images owner]
      (om/component
        (dom/div #js {:style #js {:background-color "#8a6d3b" :padding 5}}
          (om/build image-grid images {:opts {:compo image-300-compo
                                              :max-height 300
                                              :spacing 10}}))))
    (->> images-300 shuffle (take 10))))

(def images-400
  [{:file "400/01.jpg" :width 280 :height 400} {:file "400/02.jpg" :width 266 :height 400}
   {:file "400/03.jpg" :width 265 :height 400} {:file "400/04.jpg" :width 812 :height 400}
   {:file "400/05.jpg" :width 300 :height 400} {:file "400/06.jpg" :width 303 :height 400}
   {:file "400/07.jpg" :width 272 :height 400} {:file "400/08.jpg" :width 320 :height 400}
   {:file "400/09.jpg" :width 279 :height 400} {:file "400/10.jpg" :width 533 :height 400}
   {:file "400/11.jpg" :width 600 :height 400} {:file "400/12.jpg" :width 309 :height 400}
   {:file "400/13.jpg" :width 309 :height 400} {:file "400/14.jpg" :width 270 :height 400}
   {:file "400/15.jpg" :width 281 :height 400} {:file "400/16.jpg" :width 296 :height 400}
   {:file "400/17.jpg" :width 601 :height 400} {:file "400/18.jpg" :width 295 :height 400}
   {:file "400/19.jpg" :width 317 :height 400} {:file "400/20.jpg" :width 252 :height 400}
   ])

(defn image-400-compo [{:keys [file width height]} owner]
  (reify
    om/IRenderState
    (render-state [_ {:keys [grid-width grid-height]}]
      (dom/div #js {:style #js {:display "inline-block"}}
        (dom/img #js {:src file :width (or grid-width width) :height (or grid-height height)})))))

(defcard images-400-px
  (dc/om-root-card
    (fn [images owner]
      (om/component
        (dom/div #js {:style #js {:background-color "#999" :padding 10}}
          (om/build image-grid images {:opts {:compo image-400-compo
                                              :max-height 400
                                              :spacing 0}}))))
    (->> images-400 shuffle (take 10))))
