(ns infer.core
  (:require [clojure.string]))

(defrecord Routes [routes params])

(defprotocol IMatchRoutes
  (match-route [_ p]))

(defprotocol ICompilePaths
  (compile-path [_ r p] [_ r]))

(defn create-route
  "Create route from rule"
  [rule]
  (assert (string? rule) "Route rule should be a string")
  (assert (= (first rule) \/) "Route rules need forward slash")
  (assert (not= (-> rule rest last) \/) "Route rules do not need ending slash")
  (let [split (-> rule (subs 1) (clojure.string/split #"/"))]
    (assert (not (some #{""} split)) "Route rule has consecutive forward slashes")
    (mapv #(if (= (first %) \:)
             (do (assert (> (count %) 1) (str "Bad key for route " rule))
                 (keyword (subs % 1)))
             %) split)))

(defn create-routeset
  "Create routeset from rule"
  [rules]
  (reduce #(let [route  (create-route %2)
                 params (->> route (filter keyword?) set)]
             (assert (not (some #{params} (-> %1 keys set))) (str "Route with params " params " already exists"))
             (assoc %1 params route)) {} rules))

(defn create-routes
  "Create routes from user rules"
  ([rules params]
   (let [routes (zipmap (keys rules) (map #(if (vector? %)
                                            (create-routeset %)
                                            (create-route %))
                                          (vals rules)))]
     (Routes. routes params)))
  ([rules]
   (create-routes rules {})))

(defn route-matches
  "Match path against a route"
  [route path]
  (let [path-split (clojure.string/split (subs path 1) #"/")]
    (when (= (count path-split) (count route))
      ((fn step [value route path]
         (let [route-c (first route)
               path-c (first path)
               routes-r (rest route)
               paths-r (rest path)]
           (if (keyword? route-c)
             (let [value' (assoc value route-c path-c)]
               (if (not-empty routes-r)
                 (step value' routes-r paths-r)
                 value'))
             (if (= route-c path-c)
               (if (not-empty routes-r)
                 (step value routes-r paths-r)
                 value)
               nil))))
       {} route path-split))))

(defn apply-lizers [params lizers]
  "Apply function by key to parameter map"
  (reduce (fn [r [k v]]
            (assoc r k (if-let [lizer (get lizers k)]
                         (lizer v)
                         v)))
          {} params))

(defn get-route [routes path]
  "Get route and params"
  ((fn step [routes]
     (let [route      (first routes)
           routes-r   (rest routes)
           route-name (first route)
           route-rule (second route)]
       (if-let [params (if (map? route-rule)
                         (->> (vals route-rule)
                              (filter vector?)
                              (map #(route-matches % path))
                              (filter #(not (nil? %)))
                              first)
                         (route-matches route-rule path))]
         [route-name params]
         (when (not-empty routes-r)
           (step routes-r)))))
   (lazy-seq routes)))

(extend-type Routes
  IMatchRoutes
  (match-route [this path]
    (if-let [[route-name route-params] (-> this :routes (get-route path))]
      [route-name (apply-lizers route-params (-> this :params :deserialize))]
      [:not-found {:path path}])))

(defn route-to-path
  "Create path from route and params"
  ([route params]
   (->> (mapv #(if (keyword? %)
                 (let [p (get params %)]
                   (assert p (str "Need param " % " for " route))
                   p)
                 %) route)
        (interpose "/")
        (cons \/)
        (apply str)))
  ([route]
   (route-to-path route {})))

(defn create-path
  "Create path by route name and parameters given"
  ([routes route-name params]
   (if-let [route (get routes route-name)]
     (if (map? route)
       (let [route  (->> params keys set (get route))]
         (assert route (str "No route for " route-name " with params " params))
         (route-to-path route params))
       (route-to-path route params))
     (assert false (str "No route for " route-name))))
  ([routes route-name]
   (create-path routes route-name {})))

(defn remove-empties [m]
  "Remove empty values from a map"
  (reduce (fn [r [k v]]
            (if (not-empty v)
              (assoc r k v)
              r)) {} m))

(extend-type Routes
  ICompilePaths
  (compile-path
   ([this route-name params]
    (let [serialized (->> this :params :serialize (apply-lizers params) remove-empties)]
      (create-path (:routes this) route-name serialized)))
   ([this route-name]
    (compile-path this route-name {}))))
