# infer

Two-way routing for clojurescript

#### Warning
- No encoding/decoding stuff if that's something you are interested in
- At least currently no query param handling

### Usage
```clojure
["whodidthis/infer" "0.1.0"]
```
```clojure
(:require [infer.core :as infer])
```
#### Route matching
Infer creates routes from a map of `:route-name "/route"`, match is returned as a vector `[:route-name {:param "value"}]` or `[:not-found {:path "path"}]`.
```clojure
(def routes (infer/create-routes {:image/show "/image/:id"
                                  :front      "/"}))
(infer/match-route routes "/image/501") ;=> [:image/show {:id "501"}]
(infer/match-route routes "/") ;=> [:front {}]
(infer/match-route routes "/where") ;=> [:not-found {:path "/where"}]
```
You can also have multiple paths per route name by wrapping them in a vector. 

```clojure
(def routes (infer/create-routes {:image/all ["/image/tagged/:tags" "/image"]
                                  :front     "/"}))
(infer/match-route routes "/image") ;=> [:image/all {}]
(infer/match-route routes "/image/tagged/cool") ;=> [:image/all {:tags "cool"}]
```

Deserialization can be set up for each parameter when creating routes.

```clojure
(def routes (infer/create-routes {:image/all ["/image/tagged/:tags" "/image"]
                                  :front     "/"}
                                 {:deserialize {:tags #(-> (clojure.string/split % #","))}}))
(infer/match-route routes "/image/tagged/cool,nice") ;=> [:image/all {:tags ["cool" "nice"]}]
```
#### Path compilation
And paths can be compiled by giving route name and required parameters. 

```clojure
(def routes (infer/create-routes {:food-page ["/foods/:foods" "/foods"]
                                  :flower-page "/flower/:flower"}))
(infer/compile-path routes :food-page {:foods "good-foods"}) ;=> "/foods/good-foods"
(infer/compile-path routes :food-page) ;=> "/foods"
(infer/compile-path routes :flower-page {:flower "tulip"}) ;=> "/flower/tulip
(infer/compile-path routes :flower-page) ;=> Error, all the required parameters have to be not-empty
```

Serialization can be set up.
```clojure
(def routes (infer/create-routes {:food-page ["/foods/:foods" "/foods"]
                                  :flower-page "/flower/:flower"}
                                 {:serialize {:foods #(->> (interpose "+" %) (apply str))}}))
(infer/compile-path routes :food-page {:foods ["broccoli" "carrots"]}) ;=> "/foods/broccoli+carrots"
(infer/compile-path routes :food-page {:foods []}) ;=> "/foods"
```
