(ns webnf.base.uri
  #?(:clj (:import (java.net URLEncoder URLDecoder)))
  (:require [clojure.string :as str]
            [webnf.base.string :refer [into-str]]
            [webnf.base.util :refer [update! conjv]]))

#?
(:clj
 (do (defn encode-uri-component [s]
       (when s
         (URLEncoder/encode s "UTF-8")))
     (defn decode-uri-component [s]
       (when s
         (URLDecoder/decode s "UTF-8"))))
 :cljs
 (do (def encode-uri-component js/encodeURIComponent)
     (def decode-uri-component js/decodeURIComponent)))

(defn path->href
  "Joins path segments (each a seq) into an absolute url path"
  [& path-fragments]
  (str/join "/" (cons "" (map encode-uri-component
                              (apply concat path-fragments)))))

(defn href->path
  "Splits an absolute href into a path segment"
  [href]
  (case href
    ""  (throw (ex-info "Not an absolute path" {:href href}))
    "/" []
    (let [p (map decode-uri-component
                 (str/split (str/replace href "+" " ")
                            #"/"))]
      (when-not (= "" (first p))
        (throw (ex-info "Not an absolute path" {:href href})))
      (vec (rest p)))))

(def et (cons "&" nil))

(defn encode-uri-params
  "Encode a map of {\"key\" [\"values\",,,]} into a form-params string"
  [params]
  {:pre [(map? params)
         (every? string? (keys params))
         (every? coll? (vals params))
         (every? #(every? string? %) (vals params))]}
  (into-str
   ""
   (comp
    (mapcat    #(for [p (get params %)]
                  (list (encode-uri-component %) "=" (encode-uri-component p))))
    (interpose et)
    cat)
   (keys params)))

(defn decode-uri-params
  "Decode a form-params string into a map of {\"key\" [\"values\",,,]}"
  [s]
  (persistent!
   (reduce (fn [tm s]
             (let [[k v] (str/split s #"=" 2)]
               (update! tm (decode-uri-component k) conjv (decode-uri-component v))))
           (transient {})
           (str/split (str/replace s "+" " ")
                      #"&"))))
