(ns {{name}}.core
  (:require [cljs.reader :as reader]
            [goog.events :as events]
            [goog.dom :as gdom]
            [goog.Uri.QueryData :as query-data]
            [goog.structs :as structs]
            [om.core :as om :include-macros true]
            [om.dom :as dom :include-macros true])
  (:import [goog.net XhrIo]
           [goog.net.EventType]
           [goog.events EventType]))


(enable-console-print!)


(def ^:private meths
  {:get "GET"
   :put "PUT"
   :post "POST"
   :delete "DELETE"})


;"Content-Type" "application/edn"
(defn edn-xhr [{:keys [method url data on-complete]}]
  (let [xhr (XhrIo.)
        m (if method (method meths) (:get meths))
        _ (.log js/console m)
        headers (structs/Map. #js {"Accept" "application/edn"})
        d (when data (pr-str data))]
    (when on-complete
      (events/listen xhr goog.net.EventType.COMPLETE
                     (fn [e]
                       (on-complete (reader/read-string (.getResponseText xhr))))))
    (.send xhr url m d headers)
    xhr))


(def app-state (atom {:data nil
                      :error nil}))


;(reset! app-state {:data nil})

(defn query-table-view [{:keys [heading table]} owner]
  (reify
    om/IRender
    (render [this]
     (dom/div #js {:className "panel panel-default"}
      (dom/div #js {:className "panel-heading"} heading)
       (dom/div #js {:className "panel-body " }
        (dom/div #js {:className "table-responsive"}
        (dom/table #js {:className "table table-bordered "}
          (apply dom/thead nil
            (for [h (keys (first table))
                  :let [hs (str h)]]
              (dom/th nil hs)))
          (apply dom/tbody nil
            (for [r table]
              (apply dom/tr nil
                 (for [c (vals r)
                       :let [cs (pr-str c)]]
                   (dom/td nil cs)))))))  )))))



(defn query-view [data owner ]
  (reify
    om/IRender
    (render [this]
       (apply dom/div nil
          (for [t data
                :let [heading (str (key t))
                      table (val t)]]
            (dom/div nil
             (om/build query-table-view {:heading heading
                                         :table table})))))))


(defn error-view [e owner]
  (reify
    om/IRender
    (render [this]
       (dom/div #js {:className
                     (if e "alert alert-warning" nil)}
           (str e)))))


(defn slurp-data [app owner]
  (let [new-query (-> (om/get-node owner "new-query")
                        .-value)
        url (str "query?" new-query )]
    (.log js/console url)
    (edn-xhr {:url url
              :on-complete (fn [[data error]]
                              (om/transact! app :error (fn [_] error))
                              (om/transact! app :data (fn [_] data))) })
    (om/transact! app :data (fn [_] nil))))



(defn handle-change [e owner {:keys [text]}]
  (om/set-state! owner :text (.. e -target -value)))


(defn query-input [app owner]
  (reify
    om/IInitState
    (init-state [_]
      {:text ""})
    om/IRenderState
    (render-state [this state]
       (dom/div nil
          (dom/div #js {:className "input-group"}
          (dom/span #js {:className "input-group-addon"} "Query Name & params ")
          (dom/input #js {:className "form-control"
                          :type "text"
                          :placeholder "name=instance&id=12762692"
                          :ref "new-query"
                          :value (:text state)
                          :onKeyPress #(when (== (.-keyCode %) 13)
                                         (slurp-data app owner))
                          :onChange #(handle-change % owner state)}))
       (dom/br nil)))))


(defn app-view [app owner]
  (reify
    om/IInitState
    (init-state [_]
      {:text ""})
    om/IRender
    (render [this ]
      (dom/div nil
        (dom/br nil )
        (dom/br nil )
        (dom/br nil )
        (om/build query-input app)
        (om/build error-view (:error app))
        (dom/br nil )
        (om/build query-view (:data app))))))


(defn ^:export run [id]
  (om/root app-view app-state
    {:target (. js/document (getElementById id))}))























