(ns {{name}}.system
    (:require
     [clojure.java.io :as io]
     [clojure.tools.logging :as log]
     [tie.core :as tie]
     [tie.jdbc-util :as tj]
     [{{name}}.core.lifeCycle :refer :all]
     [{{name}}.http-route.routes :as r]))


(defrecord Config [file-name env]
  LifeCycle
  (start [this comp]
    (log/info (str "Loading config file..." file-name) )
    (let [app-config (-> file-name
                         (io/resource)
                         (slurp)
                         (read-string))
          default-config (:config app-config)
          config (merge default-config (get app-config env))]
      (assoc this :config config)))
  (stop [this] this))


(defrecord SqlDB []
  LifeCycle
  (start [this comp]
    (log/info "SQL Database connection is opening, please wait...")
    (let [tie-file-name (get-in comp [:config :config :db :file-name])
          tms (tie/load-tie-file tie-file-name)
          ds (tj/create-ds tms)]
      (tj/init-db tms ds)
      (-> this
          (assoc :tie tms)
          (assoc :ds ds))))
  (stop [this]
    (log/info "SQL Database connection is closeing, please wait...")
    (tj/clean-db (:tie this) (:ds this))
    (tj/close-ds (:ds this))))


(defrecord Web []
  LifeCycle
  (start [this comp]
    (log/info "Web is starting, please wait...")
    (assoc this :handler (r/handler comp)))
  (stop [this]
    (log/info "Web is stoping, please wait...")
    (assoc this :handler nil)))


(defonce component-order [:config :sqldb :web])


(defrecord SystemRecord [config sqldb web]
  LifeCycle
  (start [this comp]
    (log/info "System is starting, please wait...")
    (reduce (fn [system key]
              (update-in system [key] start system))
            this
            component-order))
  (stop [this]
    (log/info "System is stoping, please wait...")
    (reduce (fn [system key]
              (update-in system [key] stop))
            this
            (reverse component-order))))


(defn system
  "Returns a new instance of the whole application.
   Default env :prod in config file. "
  ([] (system :prod))
  ([env]
     (->SystemRecord
      (->Config "config.edn" env)
      (->SqlDB)
      (->Web))))


(defonce handler nil)
(defonce the-system (atom nil))


(defn init
  "Init application, set web handler"
  ([] (init (start (system) nil)))
  ([s]
     (reset! the-system s)
     (alter-var-root #'handler (constantly (get-in @the-system [:web :handler])))
     (log/info "App start successfully")))


(defn destroy
  "Clean application resource"
  ([] (destroy true))
  ([destroy?]
     (log/info "System is going to down ")
     (if destroy?
       (do
         (log/info "Stoping service...")
         (stop @the-system)
         (reset! the-system nil)
         (alter-var-root #'handler (constantly nil))))))
