(ns {{name}}.http-route.routes
    (:require
     [ring.middleware.params :refer (wrap-params)]
     [ring.middleware.format-params :refer (wrap-restful-params)]
     [ring.middleware.format-response :refer (wrap-format-response wrap-restful-response wrap-json-response)]
     [compojure.core :refer :all]
     [ring.util.servlet :refer :all]
     [ring.middleware.multipart-params :refer :all]
     [clojure.tools.logging :as log]
     [ring.util.response :as resp]
     [noir.util.middleware :as noir-mid]
     [compojure.route :as route]
     [compojure.response :as response]
     [cemerick.friend :as friend]
     (cemerick.friend [workflows :as workflows]
                      [credentials :as creds])
     [{{name}}.http-route.view.template :as v]
     [{{name}}.http-route.service-routes :as s]))


(defn- warp-allow-cross-origin
  "middleware function to allow crosss origin"
  [handler]
  (fn [request]
   (let [response (handler request)]
    (assoc-in response [:headers "Access-Control-Allow-Origin"]
         "*"))))

(defn- wrap-request-log
  "Log http request"
  [app]
  (fn [{:keys [request-method uri] :as req}]
    (let [resp (app req)]
      (log/info (-> req
                    (dissoc :body)
                    (dissoc :credential-fn)
                    (dissoc :cemerick.friend/auth-config)
                    (dissoc :unauthenticated-handler)
                    (dissoc :headers)))
      resp)))


(defn- wrap-exceptions
  "Catch all system exception "
  [app]
  (fn [request]
    (try
      (app request)
      (catch Exception e
        (log/error e "System got Exception")
        {:status 500
         :headers {"Content-Type" "text/html"}
         :body "Exception in server"}))))


(defn- warp-auth
  "For authentication system "
  [app app-context]
  (let [u {:username "admin"
           :password (creds/hash-bcrypt "admin")}
        credential-fn (fn [v]
                        (if (= (:username u ) v)
                          u))]
    (friend/authenticate app
                         {:credential-fn (partial creds/bcrypt-credential-fn credential-fn)
                          :workflows [(workflows/interactive-form)]
                          :login-uri "/login"
                          :unauthorized-redirect-uri "/login"
                          :default-landing-uri "/" })))


(defn- app-routes
  "Routes for application. "
  [app-context]
  (routes
    (GET "/" [](friend/authenticated (resp/response (v/index {:title "Index page"}))))
    (GET "/ping" [](resp/response "Ping success"))
    (route/resources "/")
    (GET "/login" [:as {params :params} ]
                    (if (friend/current-authentication)
                      (resp/redirect "/")
                      (v/login "Login" params)))
    (friend/logout (ANY "/logout" request (resp/redirect "/login")))
    (context "/query" [] (s/query-routes app-context))
    (context "/user" [] (friend/authenticated (s/user-routes app-context)))
    (route/not-found "Not Found")))


(defn handler
  "Apply middleware "
  [app-context]
  (let [r (->
           (app-routes app-context)
           (wrap-params)
           (warp-auth app-context)
           (wrap-restful-params)
           (wrap-restful-response)
           (wrap-exceptions)
           (warp-allow-cross-origin)
           (wrap-request-log))]
    (noir-mid/war-handler (noir-mid/app-handler
                           [r]
                           :middleware []))))


