(ns {{name}}.core.db-util
    (:require
     [clojure.java.io :as io]))


(defn bind-error [f [val err]]
  (if (nil? err)
    (f val)
    [nil err]))


(defmacro err->> [val & fns]
  (let [fns (for [f fns] `({{name}}.core.db-util/bind-error ~f))]
    `(->> [~val nil]
          ~@fns)))


(defn load-db-file
  "Load file from class path and return as map. It will remove single-line, multi-line comments and \n from file.

   @param {string} file-name \"Name of the file.\"
   @return {map} \" Return map \""
  [file-name]
  (let [sql-str (->
                 file-name
                 (io/resource)
                 (slurp)
                 (clojure.string/replace #"--|/\*|\*/|\n" ""))
        sql-str-list (str "(" sql-str ")")
        sql-list (read-string sql-str-list)]
    (reduce (fn [v c]
              (assoc v (keyword (:name c)) c))
            {} sql-list)))


(defn load-default-params
  "Return rename key

  @params {map} db \"Sql db file\"
  @param {string|keyword} name \"Name of the query.\"
  @param {map} \"Rename map.\""
  [db name]
  {:pre [(keyword? name)]}
  (get-in db [name :default-params]))


(defn load-rename-keys
  "Return rename key

  @params {map} db \"Sql db file\"
  @param {keyword} name \"Name of the query.\"
  @param {map} \"Rename map.\""
  [db name]
  {:pre [(keyword? name)]}
  (get-in db [name :rename-keys]))


(defn sql
  "Retun query string

  @param {map} db \"Sql db file\"
  @param {keyword} name \"Name of the query.\"
  @return {string} \"Query string \"   "
  [db name]
  {:pre [(keyword? name)]}
  (if-let [q (get-in db [name :sql])]
    (let [q-list (clojure.string/split (clojure.string/trim q) #";")]
      (if (not (every? empty? q-list))
        [q-list nil]
        [nil (format "Could not find %s sql statement" name ) ]))
    [nil (format "Could not find %s sql statement" name ) ]))



(defn replace-params
  "Apply parameter to query string. If parameter is missing it will return with error message

   @param {map} m \"Parameter of the query.\"
   @param {string} q \"Name of the query.\"
   @return {vec} \" vec as success and fails \" "
  [m q]
  (let [exp-keys (set (re-seq #"\w*:\w+" q))
        inp-keys (set (map str (keys m)))]
    (if (clojure.set/superset? inp-keys exp-keys)
      [(reduce (fn [acc [k v]]
                 (clojure.string/replace acc (str k) (str v)))
               q m) nil]
      [nil (format "Parameter missing %s" (pr-str exp-keys)) ])))


(defn map-params
  [m q]
  [(map #(replace-params m %1) q) nil])


(defn reduce-params
  [q]
  (reduce (fn [[v1 e1] [v2 e2]]
            (if (and v1 v2)
              [(conj v1 v2) nil]
              (if v1
                [v2 e2]
                [v1 e1])))
          [[] nil]
          q))



(defn sql-with-params
  "Retun query string with parameters

  @param {map} db \"Sql db file\"
  @param {keyword} name \"Name of the query.\"
  @param {map} params \"Parameter of the query.\"
  @return {string} \"Query string \"   "
  [db name & [params]]
  {:pre [(keyword? name)]}
  (let [dp (load-default-params db name)
        params (if dp
                 (merge dp params)
                 params)]
    (err->> name
            (partial sql db)
            (partial map-params params)
            reduce-params)))


(defn rename-keys [[data error] db name ]
  {:pre [(keyword? name)]}
  (if data
    (let [new-keys (load-rename-keys db name)]
      [(->> data
           (map (fn [v]
                  (clojure.set/rename-keys v new-keys)))) nil] )
    [data error]))


(defn run-query [[q error] executor ]
  (if q
    [(->> q
          (map #(executor %1 ))
          (flatten)) nil]
    [q error]))



#_(defn rename
  "Rename query name using rename-map

  @param {map} m \"Rename map\"
  @param {keyword} name \"Name of the query.\"
  @param {map} params \"Parameter of the query.\"
  @return {vec} \" Return as [name params] format\""
  [m name params  ]
  {:pre [(keyword? name)]}
  (let [name (if-let [v (get m name)] v name)]
    [name (clojure.set/rename-keys params m)]))


