(ns user
  (:require
   [ring.server.standalone :refer :all]
   [clojure.pprint :refer :all]
   [clojure.java.io :as io]
   [clojure.string :as str]
   [clojure.pprint :refer (pprint)]
   [clojure.repl :refer :all]
   [clojure.test :as test]
   [clojure.tools.namespace.repl :refer (refresh refresh-all)]
   [dev :as dev]
   [{{name}}.system :as sys]
   [{{name}}.core.lifeCycle :refer :all]))


(defonce the-system nil)

(defn- update-system [state & [c]]
  (condp = state
    :init
    (alter-var-root #'the-system (constantly c))
    :start
    (alter-var-root #'the-system (fn [s]
                                   (when s
                                     (if c
                                       (assoc s c (start (c s) s))
                                       (start s nil)))))
    :stop
    (alter-var-root #'the-system (fn [s]
                                   (when s
                                     (if c
                                       (assoc s c (stop (c s)))
                                       (stop s)))))))


(defn app
  "repl command for component life cycle.

   @param  {keyword} state \"State of the application- init, start, stop\"
   @param? {keyword} opt \"Comp of the application\"

   Example
   (app :init)            \" Init application \"
   (app :start)           \" Start application \"
   (app :go)              \" init then start \"
   (app :reset)           \" Stop application and reload namespace \"
   (app :restart comp)  \" Stop application and restart component, need to reload namespace manually \""
  ([] (app :go))
  ([state & [opt]]
     (println (str "Running, please wait.." state) )
     (condp = state
       :init
       (let [server (dev/->Server 8083)
             s (-> (sys/system :dev)
                   (assoc :server server)) ]
         (update-system :init s))

       :start
       (if opt
         (update-system :start opt)
         (do
           (update-system :start)
           (update-system :start :server)))

       :go
       (if the-system
         (println "System already running")
         (do
           (app :init)
           (app :start)))

       :stop
       (if opt
         (update-system :stop opt)
         (do
           (update-system :stop)
           (update-system :stop :server)))

       :restart
       (let [c-list (drop-while #(not (= % opt)) sys/component-order)]
         (if-not (empty? c-list)
           (do
             (doseq [c c-list] (update-system :stop c))
             (doseq [c c-list] (update-system :start c)))
         (println "Component not found")))

       :reset
       (do
         (app :stop)
         (println "App stop done")
         (refresh :after 'user/app))
;:after 'user/app
       :help
       (pprint {:init "Init component"
                :start "Start all component, "
                :go "Init with start"
                :stop "Stop all component"
                :restart "Restart component"
                :reset "Reset application "})
       (println "Command not found"))

     state))
