(ns {{name}}.modules.user.auth-handler
   (:use
     [metis.core ]
     [datomic.api :only [q db] :as d])
   (:require
     [compojure.core :refer :all]
     [ring.util.response :as resp]
     [clojure.tools.logging :as log]
     (cemerick.friend [credentials :as creds])))


(defvalidator user-validator
  [[:lname :sname] :presence {:message "This field is required."} ]
  [:lname :with {:validator (fn [attr] true) :message "error!"}]
  [[:lname :sname] :formatted {:pattern #"[A-Za-z]+" :message "wrong formatting!"}])



#_(defn from-view-model [{:keys [id sname lname]}]
    {:db/id id
     :user/password (creds/hash-bcrypt sname)
     :user/l-name lname
     :user/s-name sname })



(defn to-view-model [db-entity ]
  {:id (:user/s-name db-entity)
   :lname (:user/l-name db-entity)
   :sname (:user/s-name db-entity)
   :password (:user/password db-entity)})



(defn- find-entity-by-name [conn name]
  (let [rule '[[(active? ?u)
                [?u :user/status :user.status/active]]]
        q '[:find ?u
            :in $ ?name %
            :where
            [?u :user/s-name ?name]
            (active? ?u)]]
    (ffirst (d/q q (d/db conn) name rule))))



(defn- find-all-entity [conn]
  (let [rule '[[(active? ?u)
                [?u :user/status :user.status/active]]]
        q '[:find ?u
            :in $ %
            :where
            [?u :user/s-name]
            (active? ?u)]]
    (d/q q (d/db conn) rule)))



(defn insert-all [conn data]
  (let [tempid (d/tempid :db.part/db )
        data (conj data {:db/id tempid
                         :user/s-name "admin"
                         :user/l-name "admin"
                         :user/password (creds/hash-bcrypt "admin")
                         :user/status :user.status/active}) ]
    @(d/transact conn data)
    nil))


(defn find-all [context page]
  (log/info "Getting user list")
  (let [size 15
        page (Integer/parseInt (or page "1"))
        conn (get-in context [:datomic :conn])
        p (partition-all size (find-all-entity conn))
        v (nth p (- page 1) (last p) ) ]
    (->> v
         (map (fn [r]
                (-> (d/db conn)
                    (d/entity (first r))
                    (d/touch)
                    (to-view-model)))))))


(defn details [context name]
  (let [conn (get-in context [:datomic :conn])
        entity-id (find-entity-by-name conn name)]
    (if entity-id
      (-> (d/db conn)
          (d/entity entity-id)
          (d/touch)
          (to-view-model)))))


(defn create! [context params]
  (let [error-param (user-validator params)]
    (if (empty? error-param)
      (let [conn (get-in context [:datomic :conn])
            tempid  (second  (second  (d/tempid :db.part/db )))
            v [{:db/id tempid
                :user/l-name (:lname params)
                :user/s-name (:sname params)
                :user/status :user.status/active}]]
        @(d/transact conn  v)
        nil)
      error-param)))


(defn update! [context name params]
  (let [error-param (user-validator params)]
    (if (empty? error-param)
      (let [conn (get-in context [:datomic :conn])
            v [{:db/id (find-entity-by-name conn name)
                :user/l-name (:lname params)
                :user/s-name (:sname params)
                :user/status :user.status/active}]]
        @(d/transact conn v)
        nil)
      error-param )))


(defn delete! [context user]
  (try
    (let [conn ((get-in context [:datomic :conn]))
          v [{:db/id (find-entity-by-name conn name)
            :user/status :user.status/delete}] ]
      @(d/transact conn v)
      {:success true})
      (catch Exception e
        {:error (.getMessage e)})))


#_(defn auth-validation [app-context user]
  (when-let [d-user (details app-context user )]
        {:username (:s-name  d-user)
         :password (:password d-user)}))


(defn auth-routes
  "Routes for autherization handler

  @param {map} app-context \"App context map generated by system init\"
  @return {compojure.core.routes}"
  [app-context]
  (routes
    (GET "/"  [page :as r] (resp/response (find-all app-context page)))
    (POST "/" [:as {:keys [params]}] (resp/response (create! app-context params )))
    (context "/:id" [id]
             (routes
               (GET "/" [] (resp/response (details app-context id)))
               (DELETE "/" [](resp/response (delete! app-context id)))
               (POST "/" [:as {:keys [params]}](resp/response (update! app-context id params )))))))



(comment
  (use 'dev)

  (create (:datomic-conn  the-system) {:lname "mamun" :sname "mamunabc"})
  (println (find-all (:datomic-conn  the-system) 1 5))
  (println (find-by-name (:datomic-conn  the-system) "mamunabc"))


  (if-let [entity-id (find-entity-by-name (:datomic-conn the-system) "mamunabc")]
    (println  (-> (d/db (:datomic-conn the-system))
                  (d/entity entity-id)
                  (d/touch))))

  )





