(ns {{name}}.modules.query-handler
    (:use
     [compojure.core])
    (:require
      [ring.util.response :as resp]
      [clojure.tools.logging :as log]
      [clojure.java.jdbc :as jdbc]
      [{{name}}.core.db-util :as db-util]))



(defmulti ^:private execute-query
  "Query execution method"
  (fn [v] (:name v) ))


(defmethod ^:private execute-query :table [{:keys [conn db name params]}]
  (if-let [model (:tname params)]
    (let [[q error] (db-util/get-sql-with-params db :table params)]
        (if q
          (jdbc/query conn [q] :as-arrays? true )
          error ))
    {:error {:des "Provide table-name parameter"}}))


(defmethod ^:private execute-query :table-by-id [{:keys [conn db name params]}]
  (if-let [model (:tname params)]
    (let [[q error] (db-util/get-sql-with-params db :table-by-id params)]
        (if q
          (jdbc/query conn [q] )
          error ))
    {:error {:des "Provide table-name parameter"}}))



(defmethod ^:private execute-query :users [{:keys [conn db name params]}]
  (let [[q error] (db-util/get-sql-with-params db :users params)]
    (if q
      (jdbc/query conn [q] :as-arrays? true )
      error )))


(defmethod ^:private execute-query :default [{:keys [conn db name params]}]
  (if (nil? name)
    {:error {:des "Provide name parameter"}}
    (let [[q error] (db-util/get-sql-with-params db name params)]
      (if q
        (jdbc/query conn [q])
        error ))))


(def ^:dynamic *rename-map*
  "Dynamic rename keyword. It should not contain duplicate key. Please check clojure.set/rename-keys doc"
  {:data :table
   :data-id :table-by-id
   :model :tname
   :user_id :user-by-id})


(defn handler
  "Default handler for query execution. Rename {name, parameter} using *rename-map* then execute query.

   @param {map} app-context \"App context must contain [:sqldb :conn] for database connection.
                             and [:sqldb :db] for query file.\"
   @param {string | keyword} name \"Name of the query.\"
   @param {map} params \"Parameter of the query.\"
   @return {vec} \" First -> data, second -> error. On of them will be null [data nil] or [nil data]\""
  [app-context name params]
  (try
    (let [name (if (and (= name "data") (get params :id))
                   "data-id"
                   name)
          [name params] (db-util/rename *rename-map* name params )
          db @(get-in app-context [:sqldb :db])
          conn @(get-in app-context [:sqldb :conn])]
      (execute-query {:name name
                      :params params
                      :conn conn
                      :db db}))
    (catch Exception e
      (do
        (log/info (format "sql statement failed name: %s params: %s" name (pr-str params)))
        (log/error e)
        "Query operation fail, please check log "))))




(defn query-routes
  "Routes for query_handler

   @param {map} app-context \"App context map generated by system init\"
   @return {compojure.core.routes}"
  [app-context]
  (routes
    (GET "/" [:as {:keys [params]}]
         (resp/response (handler app-context (:name params) params)))
    (context "/:name" [name]
             (routes
               (GET "/" [:as {:keys [params]}]
                    (resp/response (handler app-context name params)))))))
