(ns {{name}}.handler
    (:use
     [ring.util.servlet]
     [ring.middleware.multipart-params]
     [ring.middleware.params :only [wrap-params]]
     [ring.middleware.format-params :only [wrap-restful-params]]
     [ring.middleware.format-response :only [wrap-format-response wrap-restful-response wrap-json-response]]
     [compojure.core]
     [cljs.net.remote.service]
     [{{name}}.core.lifeCycle])
    (:require
     [clojure.tools.logging :as log]
     [ring.util.response :as resp]
     [noir.util.middleware :as noir-mid]
     [compojure.route :as route]
     [compojure.response :as response]
     [cemerick.friend :as friend]
     [net.cgrand.enlive-html :as html]
     [{{name}}.modules.query-handler :as query]
     [{{name}}.modules.remote-handler :as remote]
     [{{name}}.modules.user.auth-handler :as auth]
     [{{name}}.modules.user.user-handler :as user]
     [{{name}}.modules.home.home-handler :as home]
     [{{name}}.modules.upload.upload-handler :as upload]
     [{{name}}.modules.reporting.report :as report]))


(defn wrap-request-log [app]
  (fn [{:keys [request-method uri] :as req}]
    (let [resp (app req)
          _ (log/info (-> req
                          (dissoc :body)
                          (dissoc :credential-fn)
                          (dissoc :cemerick.friend/auth-config)
                          (dissoc :unauthenticated-handler)
                          (dissoc :headers)))]
      resp)))


(defn wrap-exceptions [app]
  (fn [request]
    (try
      (app request)
      (catch Exception e
        (log/error e "System got Exception")
        {:status 500
         :headers {"Content-Type" "text/html"}
         :body "Exception in server"}))))



(html/deftemplate index "public/index.html" [{:keys [title]}]
  [:head :title] (html/content title)
  [:script html/any-node] (html/replace-vars {:name "world" }))


(defn ^:private handler [app-context]
  (let [v (concat [(route/resources "/")]
                  [(GET "/" [] (friend/authenticated (resp/response (apply str (index {:title "Index page"})))))]
                  [(GET "/ping" [] (resp/response "Ping success"))]
                  (query/handler app-context)
                  (remote/handler app-context)
                  (auth/handler app-context)
                  (user/handler app-context)
                  (home/handler app-context)
                  (report/handler app-context)
                  (upload/handler app-context)
                  [(route/not-found "Not Found")])
        r (->
           (apply routes v)
           (wrap-params)
           (auth/auth-warp (auth/auth-validation app-context))
           (wrap-restful-params)
           (wrap-restful-response)
           (wrap-exceptions)
           (wrap-request-log))]
    (noir-mid/war-handler (noir-mid/app-handler
                           [r]
                           :middleware []))))


(defrecord Web []
  LifeCycle
  (start [this comp]
    (log/info "Web is starting, please wait...")
    (assoc this :handler (handler comp)))
  (stop [this]
    (log/info "Web is stoping, please wait...")
    (assoc this :handler nil)))
