(ns {{name}}.system
  (:use
    [{{name}}.core.lifeCycle])
  (:require
   [clojure.java.io :as io]
   [clojure.tools.logging :as log]
   [{{name}}.handler :as h]
   [{{name}}.db :as db]))


(defrecord Config [file-name env]
  LifeCycle
  (start [this comp]
    (log/info "Start reading config file")
    (let [app-config (-> file-name
                 (io/resource)
                 (slurp)
                 (read-string))
          default-config (:config app-config)
          config (merge default-config (get app-config env))]
      (assoc this :config config)))
  (stop [this] this))


(defonce component-order [:config :sqldb :datomic :web])


(defrecord SystemRecord [config sqldb datomic web]
  LifeCycle
  (start [this comp]
    (println "System is starting..................")
    (reduce (fn [system key]
              (update-in system [key] start system))
            this
            component-order))
  (stop [this]
    (println "System is going to stop")
    (reduce (fn [system key]
              (update-in system [key] stop))
            this
            (reverse component-order))))


(defn system
  "Returns a new instance of the whole application. Default :prod"
  ([] (system :prod))
  ([env]
   (->SystemRecord
   (->Config "config.edn" env)
   (db/->SqlDB)
   (db/->Datomic)
   (h/->Web))))


(defonce handler nil)
(defonce the-system (atom nil))


(defn init
  "Init application, set web handler"
  ([] (init (start (system) nil)))
  ([s]
     (reset! the-system s)
     (alter-var-root #'handler (constantly (get-in @the-system [:web :handler])))
     (log/info "App start successfully")))


(defn destroy
  "Clean application resource"
  ([] (destroy true))
  ([destroy?]
     (log/info "System is going to down ")
     (if destroy?
       (do
         (log/info "Stoping service...")
         (stop @the-system)
         (reset! the-system nil)
         (alter-var-root #'handler (constantly nil))))))
