(ns jax.impl.runtime
  (:require [clojure.core.async :as async]
            [clojure.core.async.impl.protocols :as async.proto]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [jax.patcher :as patcher]
            [jax.patcher.compiler.max :as compiler]
            [jax.impl.patch :as patch]
            [jax.impl.system :as system]
            [jax.impl.serdes :as serdes]
            [expound.alpha :as expound])
  (:import (java.io File)
           (clojure.lang Keyword IDeref IPending)))

(defn bang
  "Send a bang message from the outlet of the jax mxj object"
  []
  (.doBang (system/jax)))

(defprotocol Inlet
  (inlet [value] [type args] "Sends a message to the inlet of the jax mxj object"))

(extend-protocol Inlet
  Double
  (inlet [value]
    (.doInlet (system/jax) value))

  Float
  (inlet [value]
    (.doInlet (system/jax) value))

  Integer
  (inlet [value]
    (.doInlet (system/jax) value))

  ^long
  (inlet [value]
    (.doInlet (system/jax) value))

  String
  (inlet
    ([value]
     (.doInlet (system/jax) value))
    ([value args]
     (.doInlet (system/jax) value (serdes/serialize args))))

  Keyword
  (inlet
    ([value]
     (.doInlet (system/jax) (name value)))
    ([value args]
     (.doInlet (system/jax) (name value) (serdes/serialize args)))))

(defprotocol Outlet
  (outlet [value] [type args] "Sends a message from the outlet of the jax mxj object"))

(extend-protocol Outlet
  Double
  (outlet [value]
    (.doOutlet (system/jax) value))

  Float
  (outlet [value]
    (.doOutlet (system/jax) value))

  Integer
  (outlet [value]
    (.doOutlet (system/jax) value))

  Short
  (outlet [value]
    (.doOutlet (system/jax) value))

  ^long
  (outlet [value]
    (.doOutlet (system/jax) value))

  String
  (outlet
    ([value]
     (.doOutlet (system/jax) value))
    ([value args]
     (.doOutlet (system/jax) value (serdes/serialize args))))

  Keyword
  (outlet
    ([value]
     (.doOutlet (system/jax) (name value)))
    ([value args]
     (.doOutlet (system/jax) (name value) (serdes/serialize args))))

  Object
  (outlet [value]
   ;; Else, fallback to reflection
    (.doOutlet (system/jax) value)))

(defn ^:no-doc eval*
  "LOW LEVEL. Send a message to thispatcher object
  See: https://docs.cycling74.com/max8/refpages/thispatcher"
  [& args]
  (outlet "eval" args))

(def ^:no-doc o (Object.))

(defprotocol JaxIdentity
  (-id [this]))

(extend-protocol JaxIdentity
  String
  (-id [this] this)

  Object
  (-id [this]
    (get this :id)))

(defn- clear-state!
  [f]
  (let [{:keys [id evaled env] :as obj} (patch/slurp-edn f)
        nodes       (patcher/->nodes (:nodes evaled))
        router-id   (:router-id env)
        total-nodes (if router-id
                      (inc (count nodes))
                      (count nodes))]

    (doseq [node nodes]
      (eval* "script" "delete" (:id node)))

    ;; TODO: why can't the router be deleted? typedmess exception thrown by thispatcher???
    (comment
     (when router-id
       (eval* "script" "delete" (str router-id))))

    (system/broadcast! [:jax/clear obj])
    (log/infof "Deleted %s (%s): %s objects" (:type evaled) id total-nodes)))

(defn clear
  "Clears the specified jax object (instrument, capture etc) from the Max runtime. If no arguments are passed, clears all objects"
  ([]
   (locking o
     (let [instance-id    (patch/instance-id)
           instance-dir   (io/file (patch/tmp-dir) instance-id)
           instance-files (filter (fn [^File f] (.isFile f)) (file-seq instance-dir))]
       (if (and instance-id (.exists instance-dir))
         (do (doseq [file instance-files]
               (try
                 (clear-state! file)
                 (finally
                   (io/delete-file file :silently true))))
             true)

         false))))

  ([x]
   (locking o
     (let [file (patch/state-file (-id x))]
       (if (.exists file)
         (try (clear-state! file)
              true
              (finally
                (io/delete-file file :silently true)))

         false)))))

(s/def ::evaluable
  (s/or
   :instrument ::patcher/instrument
   :midi-effect ::patcher/midi-effect))

(defn eval!
  "Compiles a patch and creates all Max objects that define an instrument, MIDI effect or audio effect (eg, something conforming to :jax.system/evaluable)"
  ([x]
   (eval! (compiler/default-env x) x))

  ([env x]
   (locking o
     (try
       (s/assert ::evaluable x)
       (let [id         (:id x)
             start      (System/currentTimeMillis)
             compiled   (compiler/compile env x)
             state-file (patch/state-file id)
             result     {:evaled   x
                         :id       id
                         :ts       (System/currentTimeMillis)
                         :compiled compiled
                         :env      env}]

         (when (.exists state-file)
           (clear-state! state-file))

         (when-let [parent (.getParentFile state-file)]
           (.mkdirs parent))

         (spit state-file (pr-str result))

         (doseq [cmd compiled]
           (apply eval* cmd))

         (let [end     (System/currentTimeMillis)
               elapsed (- end start)
               total   (inc (count (:nodes x)))]
           (log/infof "Initialized %s (%s): %s objects in %s ms" (:type x) id total elapsed)
           (system/broadcast! [:jax/eval result])
           true))

       (catch Throwable e
         (let [state-file (patch/state-file (:id x))]
           (try
             (clear-state! state-file)
             (catch Throwable _)
             (finally
               (io/delete-file state-file :silently true))))
         (throw e))))))

(defmacro definst
  [name x]
  `(def ~name
     (let [x#    ~x
           inst# (if (patcher/instrument? x#)
                   x#
                   (patcher/instrument x#))]
       (assoc inst# :id ~(str *ns* "/" name)))))

(defn eval-meta
  "Returns a map of metadata about the currently evaled object, nil if instrument has not been evaled."
  [x]
  (try (patch/slurp-edn (patch/state-file (-id x)))
       (catch Throwable _ nil)))

(defn- clear-captured!
  [id]
  (clear id)
  (when-let [ch (system/capture-chan id)]
    (async/close! ch)
    (system/remove-captured id)))

(defn- eval-captured!
  [id ch obj-set]
  (locking o
    (let [capture-state (patch/state-file id)]
      (try
        (s/assert ::patcher/object-set obj-set)
        (let [captured-obj-set (patcher/patch [obj     obj-set
                                               message ["message" [id "$1"] [[(patcher/ref obj 0)]] {0 :capture}]]
                                 {:inlet obj :outlet message :name :capture})
              captured-obj-set (update captured-obj-set :nodes patcher/->nodes)
              env              (compiler/default-env captured-obj-set)
              compiled         (compiler/compile env captured-obj-set)
              result           {:evaled   captured-obj-set
                                :compiled compiled
                                :env      env
                                :id       id
                                :ts       (System/currentTimeMillis)
                                :capture? true}]

          (when (.exists capture-state)
            (clear-state! capture-state))

          (when-let [parent (.getParentFile capture-state)]
            (.mkdirs parent))

          (system/add-captured id ch)

          (spit capture-state (pr-str result))

          (doseq [cmd compiled]
            (apply eval* cmd))

          (system/broadcast! [:jax/eval result])
          result)

        (catch Throwable e
          (try (clear-state! capture-state)
               (catch Throwable _)
               (finally
                 (io/delete-file capture-state :silently true)))
          (system/remove-captured id)
          (throw e))))))

(defn capture
  "Captures the contents of an object-set. Returns a core.async channel.

   The core.async channel will have a sliding-buffer with a default buffer-size of 10."
  ([obj-set]
   (capture obj-set 10))

  ([obj-set buffer-size]
   (s/assert ::patcher/object-set obj-set)
   (let [id (patcher/genid "capture")
         ch (async/chan (async/sliding-buffer buffer-size))]
     (eval-captured! id ch obj-set)
     (reify
       async.proto/ReadPort
       (take! [_ fn1-handler]
         (async.proto/take! ch fn1-handler))

       async.proto/Channel
       (close! [_]
         (clear-captured! id)
         nil)

       (closed? [_]
         (async.proto/closed? ch))

       JaxIdentity
       (-id [_] id)))))

(defn capture-signal
  "Captures the contents of an object-set whose value is a signal (~ msp obj). Returns a core.async channel.

   The core.async channel will have a sliding-buffer with a default buffer-size of 10."
  ([obj-set snapshot-interval]
   (capture-signal obj-set snapshot-interval 10))
  ([obj-set snapshot-interval buffer-size]
   (capture (patcher/patch [obj      obj-set
                            snapshot ["snapshot~" [snapshot-interval] [[(patcher/ref obj 0)]]]]
              {:name :capture-signal :inlet obj :outlet snapshot})
            buffer-size)))

(defn capture-instrument
  "Evaluates an instrument, and also captures the value from the :plugout object. Returns a core.async channel.

   The core.async channel will have a sliding-buffer with a default buffer-size of 10."
  ([instrument snapshot-interval]
   (capture-instrument instrument snapshot-interval 10))

  ([instrument snapshot-interval buffer-size]
   (let [[plugout-id plugout-idx] (:plugout instrument)
         plugout-obj  (patcher/node-by-id instrument plugout-id)
         snapshot-obj (patcher/patch [plugout  plugout-obj
                                      snapshot ["snapshot~" [snapshot-interval] [[(patcher/ref plugout plugout-idx)]]]
                                      message  ["message" [(:id instrument) "$1"] [(patcher/ref snapshot 0)] {0 :capture}]]
                        {:name :capture-instrument :inlet plugout :outlet message})
         instrument   (patcher/insert-node instrument snapshot-obj)
         id           (:id instrument)
         ch           (async/chan (async/sliding-buffer buffer-size))]

     (eval instrument)
     (system/add-captured id ch)

     (reify
       async.proto/ReadPort
       (take! [_ fn1-handler]
         (async.proto/take! ch fn1-handler))

       async.proto/Channel
       (close! [_]
         (clear-captured! id)
         nil)

       (closed? [_]
         (async.proto/closed? ch))

       JaxIdentity
       (-id [_] id)))))

(defn eval-midi-effect!
  "Evaluates a MIDI effect which routes messages of type route-id to midiout"
  [route-id]
  (let [route       (patcher/route route-id)
        patch       (patcher/patch [message ["message" ["$1"] [[route]]]]
                      {:name :midi-effect :inlet message :outlet message})
        midi-effect (patcher/midi-effect patch)]
    (eval midi-effect)))

(defn delay-inst
  [inst]
  (reify
    IDeref
    (deref [_]
      (or (eval-meta inst) (eval! inst)))

    IPending
    (isRealized [_]
      (some? (eval-meta inst)))))

(defmulti event-handler (fn [_curr-state _prev-state msg] (:type msg)))

(defmethod event-handler :default [state _ msg]
  (log/warn "Cannot handle event" (pr-str msg))
  state)

(defn event-loop
  ([ch]
   (event-loop {} ch))

  ([initial-state ch]
   (async/go-loop [state initial-state
                   prev-state nil]
     (when-let [msg (async/<! ch)]
       (let [next-state (try (event-handler state prev-state msg)
                             (catch Throwable e
                               (log/error "Failed to process message" (pr-str msg))
                               (.printStackTrace e)
                               state))]
         (recur next-state state))))))

(defn close-patch
  []
  (clear)
  (patch/stop!)
  (in-ns 'user)
  true)

(defn- dir->ns
  [dir]
  (symbol (.getName (io/file dir))))

(defn load-patch
  [dir]
  (when (patch/running?)
    (close-patch))
  (patch/load-patch dir)
  (in-ns (dir->ns dir))
  (use 'clojure.core)
  ((requiring-resolve 'user/prelude))
  true)

(defn new-patch
  [dir]
  (when (patch/running?)
    (close-patch))
  (patch/new-patch dir)
  (load-patch dir))

(defn restart-patch
  []
  (clear)
  (patch/restart!)
  true)

(defonce routes
  (atom {}))

(s/def :route/type
  #{:long :double :string :boolean})

(defmulti route :route/type)

(s/def :route/spec any?)

(s/def :route.ui/type
  #{:arc :toggle})

(defmulti route-ui :type)

(s/def :route.ui.arc/min nat-int?)
(s/def :route.ui.arc/max nat-int?)

(defmethod route-ui :arc [_]
  (s/keys :req-un [:route.ui/type
                   :route.ui.arc/min
                   :route.ui.arc/max]))

(s/def :route.ui.toggle/labels
  (s/tuple string? string?))

(defmethod route-ui :toggle [_]
  (s/keys :req-un [:route.ui.toggle/labels]))

(s/def :route/ui
  (s/multi-spec route-ui :type))

(s/def ::route
  (s/keys :req-un [:route/type]
          :opt-un [:route/ui
                   :route/spec]))

(defn defroute [k opts]
  (if (s/valid? ::route opts)
    (do (when (patch/running?)
          (system/broadcast! [:jax/defroute k opts]))
        (swap! routes assoc k opts)
        k)
    (throw (RuntimeException. ^String (expound/expound-str ::route opts)))))

(defn validate-route [id val]
  (when-let [def (get @routes (keyword id))]
    (let [type-spec (case (:type def)
                      :long int?
                      :double double?
                      :string (s/or :string string? :kw keyword? :symbol symbol?)
                      :boolean (s/or :one #{1} :zero #{0} :bool boolean?))]
      (when-not (s/valid? type-spec val)
        (throw (RuntimeException. ^String (expound/expound-str type-spec val))))
      (when-let [spec-def (:spec def)]
        (when-not (s/valid? spec-def val)
          (throw (RuntimeException. ^String (expound/expound-str spec-def val))))))))

(defn route!
  "Sends a message from the outlet of the jax mxj object, prefixed by a route, eg:

  (route! :osc/freq 400)"
  ([route val]
   (let [id (keyword (get route :id route))]
     (validate-route id val)
     (outlet (compiler/id->str id) vals)))

  ([route val & vals]
   (let [id   (keyword (get route :id route))
         vals (conj (vec vals) val)]
     (validate-route id vals)
     (outlet (compiler/id->str id) vals))))