(ns jax.impl.system
  (:require [clojure.core.async :as async]
            [clojure.tools.logging.impl :as logging.impl]
            [clojure.tools.logging :as log])
  (:import (jax JaxRuntime$JaxObject)
           (com.cycling74.max MaxSystem)))

(defonce ^:private capture-chans
  (atom {}))

(defonce ^:private jax-object
  (atom nil))

(defn capture-chan
  [id]
  (get-in @capture-chans [:captured id]))

(defn remove-captured
  [id]
  (swap! capture-chans update :captured #(dissoc % id))
  true)

(defn add-captured
  [id ch]
  (swap! capture-chans assoc-in [:captured id] ch)
  true)

(defn ^JaxRuntime$JaxObject jax
  []
  (if-let [jax @jax-object]
    jax
    (throw (RuntimeException. "Jax not initialized. Are you running in Max?"))))

(defn nrepl-port []
  (.getNREPLPort (jax)))

(defn jetty-port []
  (.getJettyPort (jax)))

(defn sente []
  (.getSente (jax)))

(defn sente-ch []
  (:ch-recv (sente)))

(defn broadcast!
  [event]
  (when-let [f (:broadcast-fn (sente))]
    (f event)))

(defn max-logger []
  (reify logging.impl/Logger
    (enabled? [_ _]
      true)
    (write! [_ level e message]
      (when e
        (.printStackTrace ^Throwable e))
      (condp = level
        :fatal (MaxSystem/error message)
        :error (MaxSystem/error message)
        (MaxSystem/post message)))))

(defn max-logger-factory []
  (reify logging.impl/LoggerFactory
    (name [_]
      "com.cycling74.max.MaxSystem")
    (get-logger [_ _]
      (max-logger))))

(defn set-logger-factory! []
  (alter-var-root #'log/*logger-factory* (constantly (max-logger-factory))))

(defn register-jax
  [jax]
  (reset! jax-object jax))

(defn handled-captured
  [id value]
  (when-let [ch (capture-chan id)]
    (broadcast! [:jax/capture [id value]])
    (async/put! ch {:id    id
                    :ts    (System/currentTimeMillis)
                    :value value})))