(ns jax.patcher.compiler.max
  "Max (thispatcher) compiler-"
  (:refer-clojure :exclude [compile])
  (:require [jax.patcher :as patcher]))

(defn id->str
  [match]
  (if (string? match)
    match
    (if-let [ns (namespace match)]
      (str ns "/" (name match))
      (name match))))

(defn build-router
  [{:keys [router-id handler-id]} nodes]
  (let [routes     (->> nodes
                        (reduce
                         (fn [routes {:keys [args inlets outlets]}]
                           (into routes
                                 (filter patcher/route?)
                                 (into args (mapcat identity (into inlets outlets)))))
                         #{})
                        (vec))
        routetable (->> routes
                        (map-indexed (fn [idx route] [(:id route) idx]))
                        (into {}))
        prepend-id (str router-id "_prepend")
        obj        {:type   ::patcher/object-set
                    :nodes  (into [{:object "route"
                                    :type   ::patcher/object
                                    :id     router-id
                                    :args   (map (comp id->str :id) routes)}
                                   {:type    ::patcher/object
                                    :object  "prepend"
                                    :args    ["router"]
                                    :id      prepend-id
                                    :outlets [[{:type ::patcher/ref
                                                :id   handler-id
                                                :idx  0}]]}]
                                  (map-indexed (fn [idx route]
                                                 {:type    ::patcher/object
                                                  :object  "prepend"
                                                  :id      (patcher/genid)
                                                  :args    [(-> route :id id->str)]
                                                  :inlets  [[{:type ::patcher/ref
                                                              :id   router-id
                                                              :idx  idx}]]
                                                  :outlets [[{:type ::patcher/ref
                                                              :id   prepend-id
                                                              :idx  0}]]})
                                               routes))
                    :id     router-id
                    :inlet  router-id
                    :outlet router-id}]
    [routetable obj]))

(defn build-connection
  [{:keys [router-id]} routetable type id idx connections]
  (map (fn [connection]
         (when connection
           (let [[other-id other-idx] (case (:type connection)
                                        ::patcher/route [router-id (get routetable (:id connection))]
                                        ::patcher/ref [(:id connection) (:idx connection)]
                                        ::patcher/event-channel [(:id connection) 0])]
             (case type
               :input ["script" "connect" other-id other-idx id idx]
               :output ["script" "connect" id idx other-id other-idx]))))
       connections))

(defn build-node-connections
  [env routetable {:keys [inlets outlets id]}]
  (->> (into (map-indexed (partial build-connection env routetable :input id) inlets)
             (map-indexed (partial build-connection env routetable :output id) outlets))
       (mapcat identity)
       (filter identity)))

(defn build-connections
  [env routetable nodes]
  (mapcat (partial build-node-connections env routetable) nodes))

(defn serialize-message-arg
  [arg]
  (if (patcher/route? arg)
    (if-let [init (:init arg)]
      init
      (throw (RuntimeException. (format "Invalid argument. Route %s contains no init value" (:id arg)))))
    arg))

(defn serialize-message-args
  [args]
  (mapv serialize-message-arg args))

(defn build-object-set-arg
  [node]
  (let [args (serialize-message-args (:args node))]
    [(into ["script" "newobject" (:object node)])
     ["script" "nth" (:id node) (:object node) 1]
     (into ["script" "send" (:id node) "set"] args)]))

(defn build-object
  [node]
  [(into ["script" "newobject" (:object node)] (serialize-message-args (:args node)))
   ["script" "nth" (:id node) (:object node) 1]])

(defn build-objects
  [nodes]
  (mapcat (fn [node]
            ;; TODO: something better...
            (let [obj-name (:object node)]
              (case obj-name
                ;; this is dirty...
                ("message" "number")
                (build-object-set-arg node)

                ;; majority of objects can be constructed (as we expect) via these two commands to thispatcher
                (build-object node))))
          nodes))

(defmulti compile
  "Compiles a jax object into a collection of instructions to be sent to a thispatcher obj"
  (fn [_env obj] (:type obj)))

(defn build-event-channels
  [{:keys [handler-id]} nodes]
  (let [event-channels (->> nodes
                            (mapcat :outlets)
                            (mapcat identity)
                            (filter #(= ::patcher/event-channel (:type %))))]
    (mapcat
     (fn [event-channel]
       (let [event-type (-> event-channel :event-type name)
             args       (serialize-message-args [event-type "$1" "$2"])]
         [["script" "newobject" "message"]
          ["script" "nth" (:id event-channel) "message" 1]
          (into ["script" "send" (:id event-channel) "set"] args)
          ["script" "connect" (:id event-channel) 0 handler-id 0]]))
     event-channels)))

(defn route-defaults
  [route-obj nodes]
  (let [parameters (patcher/nodes->parameters nodes)]
    (keep (fn [[id init]]
            (when init
              ["script" "send" (:id route-obj) (id->str id) init]))
          parameters)))

(defmethod compile ::patcher/instrument
  [{:keys [script-id router-id plugout-id] :as env} {:keys [nodes plugout]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (patcher/->nodes (conj nodes route-obj))]
    (concat
     (build-event-channels env nodes)
     (build-objects nodes)
     [["script" "connect" (first plugout) (second plugout) plugout-id 0]
      ;; TODO: this connection could be defined in `build-router`
      ["script" "connect" script-id 0 router-id 0]]
     (build-connections env routetable nodes)
     (route-defaults route-obj nodes))))

(defmethod compile ::patcher/midi-effect
  [{:keys [script-id router-id midiout-id] :as env} {:keys [nodes midiout]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (patcher/->nodes (conj nodes route-obj))]
    (concat
     (build-event-channels env nodes)
     (build-objects nodes)
     [["script" "connect" (first midiout) (second midiout) midiout-id 0]
      ["script" "connect" script-id 0 router-id 0]]
     (build-connections env routetable nodes)
     (route-defaults route-obj nodes))))

(defmethod compile ::patcher/object-set
  [{:keys [script-id router-id] :as env} {:keys [nodes]}]
  (let [nodes (patcher/->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (patcher/->nodes (conj nodes route-obj))]
    (concat
     (build-event-channels env nodes)
     (build-objects nodes)
     [["script" "connect" script-id 0 router-id 0]]
     (build-connections env routetable nodes)
     (route-defaults route-obj nodes))))

(defn default-env
  [inst]
  (let [inst-id (some-> inst :id name)]
    {:script-id  "mother"
     :midiin-id  "mother-midiin"
     :midiout-id "mother-midiout"
     :plugin-id  "mother-plugin"
     :plugout-id "mother-plugout"
     :patch-dict "mother-dict"
     :handler-id "mother-handler"
     :router-id  (patcher/genid)
     :inst-id    inst-id}))