(ns jax.patcher.compiler.elk
  "Eclipse Layout Kernel (ELK)"
  (:refer-clojure :exclude [compile]))

(defn deserialize-args [x]
  (if (map? x)
    (:init x)
    x))

(defn routes
  [nodes]
  (->> nodes
       (reduce
        (fn [routes {:keys [args inlets outlets]}]
          (into routes
                (filter #(= :jax.patcher/route (:type %)))
                (into args (mapcat identity (into inlets outlets)))))
        #{})
       (vec)))

(defn children
  [nodes]
  (let [routes (routes nodes)
        label  (str "{ROUTER}")
        router (when (seq routes)
                 [{:id     "Router"
                   :labels [{:text label}]
                   :height 16
                   :width  (* 8 (count label))}])]
    (into (vec router)
          (map (fn [{:keys [id object args]}]
                 (let [label (str object " " (pr-str (mapv deserialize-args args)))
                       width (* 8 (count label))]
                   {:id     id
                    :width  width
                    :height 16
                    :labels [{:text label}]})))
          nodes)))

(defn inlet->edges
  [obj-id inlet-idx inlets]
  (keep (fn [{:keys [idx id type] :as x}]
          (when x
            (case type
              :jax.patcher/ref
              {:sources [id]
               :targets [obj-id]
               :id      (str "inlet-" id "-" inlet-idx "-" obj-id "-" idx)}

              :jax.patcher/route
              {:sources ["Router"]
               :targets [obj-id]
               :id      (str "inlet-" id "-" inlet-idx "-" obj-id "-" idx)}

              nil)))
        inlets))

(defn outlet->edges
  [obj-id outlet-idx outlets]
  (keep (fn [{:keys [idx id type] :as x}]
          (when x
            (case type
              :jax.patcher/ref
              {:id      (str "outlet-" id "-" outlet-idx "-" obj-id "-" idx)
               :sources [obj-id]
               :targets [id]}

              :jax.patcher/route
              {:id      (str "outlet-" id "-" outlet-idx "-" obj-id "-" idx)
               :sources ["Router"]
               :targets [id]}

              nil)))
        outlets))

(defn edges
  [nodes]
  (mapcat (fn [{:keys [id inlets outlets]}]
            (into (->> (map-indexed (partial inlet->edges id) inlets)
                       (mapcat identity))
                  (->> (map-indexed (partial outlet->edges id) outlets)
                       (mapcat identity))))
          nodes))

(defn compile
  [{:keys [id nodes]}]
  {:id            (or id "root")
   :layoutOptions {"elk.algorithm" "layered"}
   :children      (children nodes)
   :edges         (edges nodes)})