(ns jax.patcher
  (:refer-clojure :exclude [ref object? clone])
  (:require [clojure.spec.alpha :as s]))

(defmulti ^:no-doc object-connection :type)

(s/def :object.connection.route/type
  #{::route})

(s/def :object.connection.route/id
  keyword?)

(s/def :object.connection.route/init
  (s/or :string string?
        :number number?
        :nil nil?))

(defmethod object-connection ::route [_]
  (s/keys :req-un [:object.connection.route/type
                   :object.connection.route/id]
          :opt-un [:object.connection.route/init]))

(s/def :object.connection.event-channel/type
  #{::event-channel})

(s/def :object.connection.event-channel/event-type
  string?)

;; TODO: event-channel only available to outlets -- refactor this multi-spec
(defmethod object-connection ::event-channel [_]
  (s/keys :req-un [:object.connection.event-channel/type
                   :object.connection.event-channel/event-type]))

(s/def :object.connection.ref/type
  #{::ref})

(s/def :object.connection.ref/id
  string?)

(s/def :object.connection.ref/idx
  nat-int?)

(defmethod object-connection ::ref [_]
  (s/keys :req-un [:object.connection.ref/id
                   :object.connection.ref/type
                   :object.connection.ref/idx]))

(s/def :object/connection
  (s/multi-spec object-connection :type))

(s/def :object/inlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/outlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/arg
  (s/or :string? string?
        :number? number?
        :route   (object-connection {:type ::route})))

(s/def ::parameters
  (s/map-of keyword? (s/or :string? string? :number? number?)))

(s/def :object/args
  (s/nilable (s/coll-of :object/arg)))

(s/def :object/id
  string?)

(s/def :object/object
  string?)

(s/def :object/type
  #{::object})

(s/def :object/name
  (s/nilable keyword?))

(s/def ::object
  (s/keys :req-un [:object/id
                   :object/object
                   :object/type]
          :opt-un [:object/inlets
                   :object/outlets
                   :object/args
                   :object/name]))

(s/def :object-set/type
  #{::object-set})

(s/def :object-set/node
  (s/or :object ::object
        :object-set ::object-set))

(s/def :object-set/nodes
  (s/coll-of :object-set/node))

(s/def :object-set/inlet
  string?)

(s/def :object-set/outlet
  string?)

(s/def :object-set/name
  keyword?)

(s/def ::object-set
  (s/keys :req-un [:object-set/type
                   :object-set/nodes]
          :opt-un [:object-set/inlet
                   :object-set/outlet
                   :object-set/name]))

(s/def :instrument/plugout
  (s/tuple string? nat-int?))

(s/def :instrument/nodes
  (s/coll-of ::object))

(s/def :instrument/id
  string?)

(s/def :instrument/type
  #{::instrument})

(s/def ::instrument
  (s/keys :req-un [:instrument/plugout
                   :instrument/nodes
                   :instrument/id
                   :instrument/type]))

(s/def :midi-effect/midiout
  (s/tuple string? nat-int?))

(s/def :midi-effect/nodes
  (s/coll-of ::object))

(s/def :midi-effect/id
  string?)

(s/def :midi-effect/type
  #{::midi-effect})

(s/def ::midi-effect
  (s/keys :req-un [:midi-effect/plugout
                   :midi-effect/nodes
                   :midi-effect/id
                   :midi-effect/type]))

(s/def ::nodes
  (s/coll-of ::node))

(s/def ::inst
  (s/coll-of ::nodes))

(defn ^:no-doc distinct-vec [xs]
  (vec (distinct xs)))

(defn object-set? [x]
  (s/valid? ::object-set x))

(defn object? [x]
  (s/valid? ::object x))

(defn instrument? [x]
  (s/valid? ::instrument x))

(defn midi-effect? [x]
  (s/valid? ::midi-effect x))

(defn route? [x]
  (= ::route (:type x)))

(defn ref? [x]
  (= ::ref (:type x)))

(defn route
  ([id]
   (route id nil))
  ([id init]
   {:type ::route
    :id   (keyword id)
    :init init}))

(defn ref
  [obj idx]
  {:type ::ref
   :id   (case (:type obj)
           ::object (:id obj)
           ::object-set (:outlet obj))
   :idx  idx})

(defn genid []
  (str (gensym "jaxobj")))

(defn event-channel
  [event-type]
  {:type       ::event-channel
   :id         (genid)
   :event-type (name event-type)})

(defn object-set
  "Creates an object-set from its arguments
   The first arg will be treated as the inlet, and the last arg will be treated as the outlet"
  [& objects]
  {:type   ::object-set
   :nodes  (vec objects)
   :inlet  (let [inlet (first objects)]
             (case (:type inlet)
               ::object (:id inlet)
               ::object-set (:inlet inlet)))
   :outlet (let [outlet (last objects)]
             (case (:type outlet)
               ::object (:id outlet)
               ::object-set (:outlet outlet)))})

(defn object
  "Creates an object map from its arguments"
  [& {:keys [object id args inlets outlets name]}]
  {:type    ::object
   :id      (or id (genid))
   :object  object
   :args    args
   :inlets  (vec inlets)
   :outlets (vec outlets)
   :name    name})

(defn ->nodes
  "Recursively walks over a collection of nodes (objects or object-sets), and flattens them into a collection of objects"
  [objs]
  (distinct-vec
   (reduce
    (fn [nodes obj]
      (cond
        (object-set? obj)
        (into nodes (->nodes (:nodes obj)))

        (object? obj)
        (conj nodes obj)

        (route? obj)
        nodes

        :else
        (throw (ex-info (str "Invalid node " obj) {:nodes nodes :obj obj :objs objs}))))
    []
    objs)))

(defn args-parameters
  [args params]
  (map (fn [arg]
         (if (route? arg)
           (assoc arg :init (get params (:id arg) (:init arg)))
           arg))
       args))

(defn inlets-parameters
  [inlets params]
  (map #(args-parameters % params) inlets))

(defn parameters
  [x params]
  (s/assert ::parameters params)
  (update x :nodes (fn [nodes]
                     (map (fn [node]
                            (-> node
                                (update :args args-parameters params)
                                (update :inlets inlets-parameters params)))
                          (->nodes nodes)))))

(defn nodes->parameters
  [x]
  (reduce
   (fn [params node]
     (let [arg-params   (filter route? (:args node))
           inlet-params (mapcat #(filter route? %) (:inlets node))]
       (into params (map (fn [{:keys [id init]}]
                           [id init]))
             (into arg-params inlet-params))))
   {}
   (->nodes (:nodes x))))

(defn instrument
  "Creates an instrument from an object set."
  ([obj-set]
   (instrument obj-set 0))
  ([obj-set plugout-idx]
   (s/assert ::instrument
             {:type    ::instrument
              :id      (genid)
              :nodes   (->nodes [obj-set])
              :plugout [(:outlet obj-set) plugout-idx]})))

(defn midi-effect
  "Creates a MIDI effect from an object set"
  ([obj-set]
   (midi-effect obj-set 0))
  ([obj-set midiout-idx]
   (s/assert ::midi-effect
             {:type    ::midi-effect
              :id      (genid)
              :nodes   (->nodes [obj-set])
              :midiout [(:outlet obj-set) midiout-idx]})))

(defn set-plugout
  [instrument obj-id obj-idx]
  (assoc instrument :plugout [obj-id obj-idx]))

(defn set-midiout
  [midi-effect obj-id obj-idx]
  (assoc midi-effect :midiout [obj-id obj-idx]))

(defn nodes-by-name
  "Returns all nodes matching name"
  [x name]
  ;; Preserve objecvt set names
  (let [nodes (distinct-vec (into (:nodes x) (->nodes (:nodes x))))]
    (filter #(= name (:name %)) nodes)))

(defn nodes-by-object-name
  "Returns all nodes matching an object name"
  [x name]
  (let [nodes (->nodes (:nodes x))]
    (filter #(= (:object %) name) nodes)))

(defn node-by-id
  "Lookup a node by its id"
  [x id]
  (let [nodes (->nodes (:nodes x))]
    (first (filter #(= id (:id %)) nodes))))

(defn names
  "Returns a set of all names for each node in x"
  [x]
  (set (keep :name (:nodes x))))

(defn ids
  "Returns a set of all ids for each node in x"
  [x]
  (set (keep :id (:nodes x))))

(defn update-node
  [x id k f]
  (update x :nodes (fn [nodes]
                     (mapv
                      (fn [node]
                        (if (= (:id node) id)
                          (update node k (partial f node))
                          node))
                      nodes))))

(defn update-outlets
  "Updates the outlets for an id relating relating to a :jax.patcher/object

 f is a function of: (fn [object outlets])"
  [x id f]
  (update-node x id :outlets f))

(defn update-inlets
  "Updates the inlets for an id relating relating to a :jax.patcher/object

 f is a function of: (fn [object inlets])"
  [x id f]
  (update-node x id :inlets f))

(defn update-args
  "Updates the arguments for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object args])"
  [x id f]
  (update-node x id :args f))

(defn update-object
  "Updates the object name for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object object-name])"
  [x id f]
  (update-node x id :object f))

(defn update-name
  "Updates the object's tagged name for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object name])"
  [x id f]
  (update-node x id :name f))

(defn set-object
  "Associates (eg, replaces) the object name for an id relating to a :jax.patcher/object"
  [x id object]
  (update-object x id (constantly object)))

(defn set-name
  "Associates (eg, replaces) the tagged name for an id relating to a :jax.patcher/object"
  [x id name]
  (update-name x id (constantly name)))

(defn set-args
  "Associates (eg, replaces) the arguments for an id relating to a :jax.patcher/object"
  [x id val]
  (update-object x id (constantly val)))

(defn set-outlets
  "Associates (eg, replaces) the outlets for an id relating to a :jax.patcher/object"
  [x id outlets]
  (update-outlets x id (constantly outlets)))

(defn set-inlets
  "Associates (eg, replaces) the inlets for an id relating to a :jax.patcher/object"
  [x id inlets]
  (update-inlets x id (constantly inlets)))

(defn insert-node
  "Insert a :jax.patcher/object into x"
  [x obj]
  (update x :nodes (comp ->nodes #(conj % obj))))

(defn insert-connection
  "For connections (eg, something conforming to :object/inlets or :object/outlets), insert a :object/connection object at the given index"
  [connections idx connection]
  (let [connections (vec connections)]
    (if (contains? connections idx)
      (update connections idx #(conj % connection))
      (let [n (- idx (count connections))]
        (into connections (conj (vec (take n (repeat nil))) [connection]))))))

(defn remove-connection
  "For connections (eg, something conforming to :object/inlets or :object/outlets), remove a matching :object/connection object from the given index"
  [connections idx connection]
  (let [connections (vec connections)]
    (if (contains? connections idx)
      (update connections idx (comp vec #(remove #{connection} %)))
      connections)))

(defn connect
  "Creates a connection between [outlet-id idx] -> [inlet-id idx]"
  [x [outlet-id outlet-idx] [inlet-id inlet-idx]]
  (let [inlet-obj (node-by-id x inlet-id)]
    (cond-> x
      inlet-obj (update-outlets outlet-id
                                (fn [_ outlets]
                                  (insert-connection outlets outlet-idx (ref inlet-obj inlet-idx)))))))

(defn disconnect
  "Removes the connection between [outlet-id idx] -> [inlet-id idx]"
  [x [outlet-id outlet-idx] [inlet-id inlet-idx]]
  (let [inlet-obj  (node-by-id x inlet-id)
        outlet-obj (node-by-id x outlet-id)]
    (cond-> x
      inlet-obj (update-outlets outlet-id
                                (fn [_ outlets]
                                  (remove-connection outlets outlet-idx (ref inlet-obj inlet-idx))))
      outlet-obj (update-inlets inlet-id
                                (fn [_ inlets]
                                  (remove-connection inlets inlet-idx (ref outlet-obj outlet-idx)))))))

(defn clone
  "Creates a new instance of an instrument - by walking all objects and generating new, unique IDs for each."
  [instrument]
  (-> instrument
      (assoc :id (genid))
      (update :nodes (fn [nodes]
                       (map #(assoc % :id (genid)) nodes)))))

(s/def :object-tuple/event-channel
  (s/map-of nat-int? keyword?))

(s/def ::object-tuple
  (s/cat :object :object/object
         :args (s/? :object/args)
         :inlets (s/? :object/inlets)
         :event-ch (s/? :object-tuple/event-channel)))

(s/def ::binding
  (s/or :object-set ::object-set
        :object ::object
        :object-name string?
        :tuple ::object-tuple))

(defn ^:no-doc binding->node
  [obj-name obj]
  (s/assert ::binding obj)
  (cond
    (s/valid? ::object-set obj)
    (assoc obj :name (keyword obj-name))

    (s/valid? ::object obj)
    (assoc obj :name (keyword obj-name))

    (string? obj)
    (object :object obj
            :args []
            :inlets []
            :outlets []
            :name (keyword obj-name))

    (s/valid? ::object-tuple obj)
    (let [[object-name args inlets event-channel?] obj
          outlets (when (map? event-channel?)
                    (let [i (apply max (keys event-channel?))]
                      (map (fn [idx]
                             (when-let [event-type (get event-channel? idx)]
                               [(event-channel event-type)]))
                           (range 0 (inc i)))))]
      (object :object object-name
              :args args
              :inlets inlets
              :outlets outlets
              :name (keyword obj-name)))))

(defn validate-obj-set
  [obj-set]
  (assert (node-by-id obj-set (:inlet obj-set))
          (str "inlet " (:inlet obj-set) " not present in patch"))
  (assert (node-by-id obj-set (:outlet obj-set))
          (str "outlet " (:outlet obj-set) " not present in patch"))
  ;; TODO: extra validations...
  ;; 1) every id in patch is unique
  ;; 2) every reference to an inlet or outlet is present in patch
  ;; 3) (one day) object name is a valid MAX/MSP object (+ type checking?)
  (s/assert ::object-set obj-set)
  obj-set)

(defn ^:no-doc -patch
  [curr-patch obj]
  (let [obj-meta (meta curr-patch)]
    (if (::init obj-meta)
      (let [{:keys [inlet outlet name]} curr-patch]
        (cond-> {:type  ::object-set
                 :nodes [obj]}
          inlet (assoc :inlet (case (:type inlet)
                                ::object (:id inlet)
                                ::object-set (:inlet inlet)))
          outlet (assoc :outlet (case (:type outlet)
                                  ::object (:id outlet)
                                  ::object-set (:outlet outlet)))
          name (assoc :name name)))
      (update curr-patch :nodes conj obj))))

(s/def ::patch-bindings
  (s/* (s/cat :binding (s/+ symbol?) :form (s/+ any?))))

(s/def :patch-return/inlet :object-set/node)
(s/def :patch-return/outlet :object-set/node)

(s/def ::patch-return
  (s/keys :opt-un [:patch-return/inlet
                   :patch-return/outlet
                   :object-set/name]))

(defmacro patch [bindings & body]
  (s/assert ::patch-bindings bindings)
  (let [ret (reduce
             (fn [curr-body-form [bind-form value-form]]
               `(let [res# (binding->node '~bind-form ~value-form)
                      ~bind-form res#]
                  (-patch ~curr-body-form res#)))
             `(let [res# (do ~@body)]
                (s/assert ::patch-return res#)
                (with-meta res# {::init true}))
             (reverse (partition 2 bindings)))]
    `(validate-obj-set (update ~ret :nodes (comp distinct-vec reverse)))))