(ns jax.patcher
  (:refer-clojure :exclude [ref])
  (:require [clojure.spec.alpha :as s]))

(defmulti ^:no-doc object-connection :type)

(s/def :object.connection.route/type
  #{::route})

(s/def :object.connection.route/id
  string?)

(s/def :object.connection.route/init
  (s/or :string string?
        :number number?
        :nil nil?))

(defmethod object-connection ::route [_]
  (s/keys :req-un [:object.connection.route/type
                   :object.connection.route/id]
          :opt-un [:object.connection.route/init]))

(s/def :object.connection.event-channel/type
  #{::event-channel})

(s/def :object.connection.event-channel/event-type
  string?)

;; TODO: event-channel only available to outlets -- refactor this multi-spec
(defmethod object-connection ::event-channel [_]
  (s/keys :req-un [:object.connection.event-channel/type
                   :object.connection.event-channel/event-type]))

(s/def :object.connection.ref/type
  #{::ref})

(s/def :object.connection.ref/id
  string?)

(s/def :object.connection.ref/idx
  nat-int?)

(defmethod object-connection ::ref [_]
  (s/keys :req-un [:object.connection.ref/id
                   :object.connection.ref/type
                   :object.connection.ref/idx]))

(s/def :object/connection
  (s/multi-spec object-connection :type))

(s/def :object/inlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/outlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/arg
  (s/or :string? string?
        :number? number?))

(s/def :object/args
  (s/nilable (s/coll-of :object/arg)))

(s/def :object/id
  string?)

(s/def :object/object
  string?)

(s/def :object/type
  #{::object})

(s/def :object/name
  (s/nilable keyword?))

(s/def ::object
  (s/keys :req-un [:object/id
                   :object/object
                   :object/type]
          :opt-un [:object/inlets
                   :object/outlets
                   :object/args
                   :object/name]))

(s/def :object-set/type
  #{::object-set})

(s/def :object-set/nodes
  (s/coll-of (s/or :object ::object
                   :object-set ::object-set)))

(s/def :object-set/inlet
  string?)

(s/def :object-set/outlet
  string?)

(s/def :object-set/name
  keyword?)

(s/def ::object-set
  (s/keys :req-un [:object-set/type
                   :object-set/nodes]
          :opt-un [:object-set/inlet
                   :object-set/outlet
                   :object-set/name]))

(s/def :instrument/plugout
  (s/tuple string? nat-int?))

(s/def :instrument/nodes
  (s/coll-of ::object))

(s/def :instrument/id
  string?)

(s/def :instrument/type
  #{::instrument})

(s/def ::instrument
  (s/keys :req-un [:instrument/plugout
                   :instrument/nodes
                   :instrument/id
                   :instrument/type]))

(s/def :midi-effect/midiout
  (s/tuple string? nat-int?))

(s/def :midi-effect/nodes
  (s/coll-of ::object))

(s/def :midi-effect/id
  string?)

(s/def :midi-effect/type
  #{::midi-effect})

(s/def ::midi-effect
  (s/keys :req-un [:midi-effect/plugout
                   :midi-effect/nodes
                   :midi-effect/id
                   :midi-effect/type]))


(s/def ::nodes
  (s/coll-of ::node))

(s/def ::inst
  (s/coll-of ::nodes))

;; operators need to be compiled to their verbose name
(defmulti obj-lookup identity)
(defmethod obj-lookup :default [x] x)
(defmethod obj-lookup "*~" [_] "times~")
(defmethod obj-lookup "-~" [_] "minus~")
(defmethod obj-lookup "*" [_] "times")
(defmethod obj-lookup "+" [_] "plus")

(defn reverse-obj-lookup []
  (let [ks (->> (methods obj-lookup)
                (keys)
                (remove #{:default}))]
    (into {} (map (fn [k] [(obj-lookup k) k])) ks)))

(defn object-name
  [{:keys [object]}]
  (obj-lookup object))

(defn reverse-object-name
  [{:keys [object]}]
  (get (reverse-obj-lookup) object object))

(defn route
  ([match]
   (route match nil))
  ([match init]
   {:type ::route
    :id   (name match)
    :init init}))

(defn ref
  [obj idx]
  {:type ::ref
   :id   (case (:type obj)
           ::object (:id obj)
           ::object-set (:outlet obj))
   :idx  idx})

(defn genid []
  (str (gensym "jaxobj")))

(defn event-channel
  [event-type]
  {:type       ::event-channel
   :id         (genid)
   :event-type (name event-type)})

(defn rget
  [m k]
  (if-let [init (get m k)]
    (route k init)
    (route k)))

(defn object-set
  "Creates an object-set from its arguments
   The first arg will be treated as the inlet, and the last arg will be treated as the outlet"
  [& objects]
  {:type   ::object-set
   :nodes  (vec objects)
   :inlet  (let [inlet (first objects)]
             (case (:type inlet)
               ::object (:id inlet)
               ::object-set (:inlet inlet)))
   :outlet (let [outlet (last objects)]
             (case (:type outlet)
               ::object (:id outlet)
               ::object-set (:outlet outlet)))})

(defn unpack-arg
  [arg]
  (if (and (map? arg) (= ::route (:type arg)))
    (:init arg)
    arg))

(defn unpack-args
  [args]
  (into [] (map unpack-arg) args))

(defn object
  "Creates an object map from its arguments"
  [& {:keys [object id args inlets outlets name]}]
  {:type    ::object
   :id      (or id (genid))
   :object  object
   :args    (unpack-args args)
   :inlets  (vec inlets)
   :outlets (vec outlets)
   :name    name})

(defn patch
  "Concisely defines an object-set"
  [& objects]
  (let [result (reduce
                (fn [{:keys [obj-lookup] :as state} obj]
                  (cond
                    (vector? (second obj))
                    (assoc (apply patch (second obj)) :name (first obj))

                    (map? (second obj))
                    (-> state
                        (assoc-in [:obj-lookup (first obj)] (second obj))
                        (update :objs conj (assoc (second obj) :name (first obj))))

                    :else
                    (let [[k object-name args inlets event-channel?] obj
                          inlets  (mapv (fn [connections]
                                          (mapv (fn [inlet]
                                                  (if (vector? inlet)
                                                    (let [[inlet-k inlet-idx] inlet]
                                                      (if (map? inlet-k)
                                                        inlet-k
                                                        (let [ref-obj (obj-lookup inlet-k)]
                                                          (ref ref-obj inlet-idx))))
                                                    inlet))
                                                connections))
                                        inlets)
                          outlets (when (map? event-channel?)
                                    (let [i (apply max (keys event-channel?))]
                                      (map (fn [idx]
                                             (when-let [event-type (get event-channel? idx)]
                                               [(event-channel event-type)]))
                                           (range 0 (inc i)))))
                          obj     (object :object object-name
                                          :args args
                                          :inlets inlets
                                          :outlets outlets
                                          :name k)]
                      (-> state
                          (assoc-in [:obj-lookup k] obj)
                          (update :objs conj obj)))))
                {:obj-lookup {}
                 :objs       []}
                objects)]

    (apply object-set (:objs result))))

(defn- distinct-vec [xs]
  (vec (distinct xs)))

(defn ->nodes
  "Recursively walks over a collection of nodes (objects or object-sets), and flattens them into a collection of objects"
  [objs]
  (distinct-vec
   (reduce
    (fn [nodes obj]
      (cond
        (and (map? obj) (= ::object-set (:type obj)))
        (into nodes (->nodes (:nodes obj)))

        (and (map? obj) (= ::object (:type obj)))
        (conj nodes obj)))
    []
    objs)))

(defn instrument
  "Creates an instrument from an object set."
  ([obj-set]
   (instrument obj-set 0))
  ([obj-set plugout-idx]
   {:type    ::instrument
    :id      (genid)
    :nodes   (->nodes [obj-set])
    :plugout [(:outlet obj-set) plugout-idx]}))

(defn midi-effect
  "Creates a MIDI effect from an object set"
  ([obj-set]
   (midi-effect obj-set 0))
  ([obj-set midiout-idx]
   {:type    ::midi-effect
    :id      (genid)
    :nodes   (->nodes [obj-set])
    :midiout [(:outlet obj-set) midiout-idx]}))

(defn set-plugout
  [instrument obj-id obj-idx]
  (assoc instrument :plugout [obj-id obj-idx]))

(defn set-midiout
  [midi-effect obj-id obj-idx]
  (assoc instrument :midiout [obj-id obj-idx]))

(defn nodes-by-name
  "Returns all nodes matching name"
  [x name]
  (let [nodes (:nodes x)]
    (filter #(= name (:name %)) nodes)))

(defn nodes-by-object-name
  "Returns all nodes matching an object name"
  [x name]
  (let [nodes (:nodes x)]
    (filter #(or (= name (object-name %))
                 (= name (reverse-object-name %)))
            nodes)))

(defn node-by-id
  "Lookup a node by its id"
  [x id]
  (let [nodes (:nodes x)]
    (first (filter #(= id (:id %)) nodes))))

(defn names
  "Returns a set of all names for each node in x"
  [x]
  (set (keep :name (:nodes x))))

(defn ids
  "Returns a set of all ids for each node in x"
  [x]
  (set (keep :id (:nodes x))))

(defn update-node
  [x id k f]
  (update x :nodes (fn [nodes]
                     (mapv
                      (fn [node]
                        (if (= (:id node) id)
                          (update node k (partial f node))
                          node))
                      nodes))))

(defn update-outlets
  "Updates the outlets for an id relating relating to a :jax.patcher/object

 f is a function of: (fn [object outlets])"
  [x id f]
  (update-node x id :outlets f))

(defn update-inlets
  "Updates the inlets for an id relating relating to a :jax.patcher/object

 f is a function of: (fn [object inlets])"
  [x id f]
  (update-node x id :inlets f))

(defn update-args
  "Updates the arguments for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object args])"
  [x id f]
  (update-node x id :args f))

(defn update-object
  "Updates the object name for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object object-name])"
  [x id f]
  (update-node x id :object f))

(defn update-name
  "Updates the object's tagged name for an id relating relating to a :jax.patcher/object

  f is a function of: (fn [object name])"
  [x id f]
  (update-node x id :name f))

(defn set-object
  "Associates (eg, replaces) the object name for an id relating to a :jax.patcher/object"
  [x id object]
  (update-object x id (constantly object)))

(defn set-name
  "Associates (eg, replaces) the tagged name for an id relating to a :jax.patcher/object"
  [x id name]
  (update-name x id (constantly name)))

(defn set-args
  "Associates (eg, replaces) the arguments for an id relating to a :jax.patcher/object"
  [x id val]
  (update-object x id (constantly val)))

(defn set-outlets
  "Associates (eg, replaces) the outlets for an id relating to a :jax.patcher/object"
  [x id outlets]
  (update-outlets x id (constantly outlets)))

(defn set-inlets
  "Associates (eg, replaces) the inlets for an id relating to a :jax.patcher/object"
  [x id inlets]
  (update-inlets x id (constantly inlets)))

(defn insert-node
  "Insert a :jax.patcher/object into x"
  [x obj]
  (update x :nodes (comp ->nodes #(conj % obj))))

(defn insert-connection
  "For connections (eg, something conforming to :object/inlets or :object/outlets), insert a :object/connection object at the given index"
  [connections idx connection]
  (let [connections (vec connections)]
    (if (contains? connections idx)
      (update connections idx #(conj % connection))
      (let [n (- idx (count connections))]
        (into connections (conj (vec (take n (repeat nil))) [connection]))))))

(defn remove-connection
  "For connections (eg, something conforming to :object/inlets or :object/outlets), remove a matching :object/connection object from the given index"
  [connections idx connection]
  (let [connections (vec connections)]
    (if (contains? connections idx)
      (update connections idx (comp vec #(remove #{connection} %)))
      connections)))

(defn connect
  "Removes the connection between [outlet-id idx] -> [inlet-id idx]"
  [x [outlet-id outlet-idx] [inlet-id inlet-idx]]
  (let [inlet-obj (node-by-id x inlet-id)]
    (cond-> x
      inlet-obj (update-outlets outlet-id
                                (fn [_ outlets]
                                  (insert-connection outlets outlet-idx (ref inlet-obj inlet-idx)))))))

(defn disconnect
  "Removes the connection between [outlet-id idx] -> [inlet-id idx]"
  [x [outlet-id outlet-idx] [inlet-id inlet-idx]]
  (let [inlet-obj  (node-by-id x inlet-id)
        outlet-obj (node-by-id x outlet-id)]
    (cond-> x
      inlet-obj (update-outlets outlet-id
                                (fn [_ outlets]
                                  (remove-connection outlets outlet-idx (ref inlet-obj inlet-idx))))
      outlet-obj (update-inlets inlet-id
                                (fn [_ inlets]
                                  (remove-connection inlets inlet-idx (ref outlet-obj outlet-idx)))))))

(defn clone
  "Creates a new instance of an instrument - by walking all objects and generating new, unique IDs for each."
  [instrument]
  (-> instrument
      (assoc :id (genid))
      (update :nodes (fn [nodes]
                       (map #(assoc % :id (genid)) nodes)))))