(ns jax.patcher
  (:refer-clojure :exclude [ref compile])
  (:require [clojure.spec.alpha :as s]))

(defmulti object-connection :type)

(s/def :object.connection.route/type
  #{::route})

(s/def :object.connection.route/id
  string?)

(s/def :object.connection.route/init
  (s/or :string string?
        :number number?
        :nil nil?))

(defmethod object-connection ::route [_]
  (s/keys :req-un [:object.connection.route/type
                   :object.connection.route/id]
          :opt-un [:object.connection.route/init]))

(s/def :object.connection.event-channel/type
  #{::event-channel})

(s/def :object.connection.event-channel/event-type
  string?)

;; TODO: event-channel only available to outlets -- refactor this multi-spec
(defmethod object-connection ::event-channel [_]
  (s/keys :req-un [:object.connection.event-channel/type
                   :object.connection.event-channel/event-type]))

(s/def :object.connection.ref/type
  #{::ref})

(s/def :object.connection.ref/id
  string?)

(s/def :object.connection.ref/idx
  nat-int?)

(defmethod object-connection ::ref [_]
  (s/keys :req-un [:object.connection.ref/id
                   :object.connection.ref/type
                   :object.connection.ref/idx]))

(s/def :object/connection
  (s/multi-spec object-connection :type))

(s/def :object/inlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/outlets
  (s/nilable (s/coll-of (s/nilable (s/coll-of (s/nilable :object/connection))))))

(s/def :object/arg
  (s/or :string? string?
        :number? number?))

(s/def :object/args
  (s/nilable (s/coll-of :object/arg)))

(s/def :object/id
  string?)

(s/def :object/object
  string?)

(s/def :object/type
  #{::object})

(s/def ::object
  (s/keys :req-un [:object/id
                   :object/object
                   :object/type]
          :opt-un [:object/inlets
                   :object/outlets
                   :object/args]))

(s/def :object-set/type
  #{::object-set})

(s/def :object-set/nodes
  (s/coll-of (s/or :object ::object
                   :object-set ::object-set)))

(s/def :object-set/inlet
  string?)

(s/def :object-set/outlet
  string?)

(s/def ::object-set
  (s/keys :req-un [:object-set/type
                   :object-set/nodes]
          :opt-un [:object-set/inlet
                   :object-set/outlet]))

(s/def :instrument/plugout
  (s/tuple string? nat-int?))

(s/def :instrument/nodes
  (s/coll-of ::object))

(s/def :instrument/id
  keyword?)

(s/def :instrument/type
  #{::instrument})

(s/def ::instrument
  (s/keys :req-un [:instrument/plugout
                   :instrument/nodes
                   :instrumnet/id
                   :instrument/type]))

(s/def ::nodes
  (s/coll-of ::node))

(s/def ::inst
  (s/coll-of ::nodes))

(def obj-lookup
  ;; operators need to be compiled to their verbose name
  {"*~" "times~"
   "*"  "times"
   "/"  "div"
   "+"  "plus"})

(defn object-name
  [{:keys [object]}]
  (get obj-lookup object object))

(defn route
  ([match]
   (route match nil))
  ([match init]
   {:type ::route
    :id   (name match)
    :init init}))

(defn ref
  [obj idx]
  {:type ::ref
   :id   (case (:type obj)
           ::object (:id obj)
           ::object-set (:outlet obj))
   :idx  idx})

(defn genid []
  (str (gensym "jaxobj")))

(defn event-channel
  [event-type]
  {:type       ::event-channel
   :id         (genid)
   :event-type (name event-type)})

(defn rget
  [m k]
  (when-let [init (get m k)]
    (route k init)))

(defn object-set
  "Creates an object-set from its arguments
   The first arg will be treated as the inlet, and the last arg will be treated as the outlet"
  [& objects]
  {:type   ::object-set
   :nodes  (vec objects)
   :inlet  (let [inlet (first objects)]
             (case (:type inlet)
               ::object (:id inlet)
               ::object-set (:inlet inlet)))
   :outlet (let [outlet (last objects)]
             (case (:type outlet)
               ::object (:id outlet)
               ::object-set (:outlet outlet)))})

(defn unpack-arg
  [arg]
  (if (and (map? arg) (= ::route (:type arg)))
    (:init arg)
    arg))

(defn unpack-args
  [args]
  (into [] (map unpack-arg) args))

(defn object
  "Creates an object map from its arguments"
  [& {:keys [object id args inlets outlets]}]
  {:type    ::object
   :id      (or id (genid))
   :object  object
   :args    (unpack-args args)
   :inlets  (vec inlets)
   :outlets (vec outlets)})

(defn patch
  "Concisely defines an object-set"
  [& objects]
  (let [result (reduce
                (fn [{:keys [obj-lookup] :as state} obj]
                  (cond
                    (vector? (second obj))
                    (apply patch (second obj))

                    (map? (second obj))
                    (-> state
                        (assoc-in [:obj-lookup (first obj)] (second obj))
                        (update :objs conj (second obj)))

                    :else
                    (let [[k object-name args inlets event-channel?] obj
                          inlets  (mapv (fn [connections]
                                          (mapv (fn [inlet]
                                                  (if (vector? inlet)
                                                    (let [[inlet-k inlet-idx] inlet]
                                                      (if (map? inlet-k)
                                                        inlet-k
                                                        (let [ref-obj (get obj-lookup inlet-k)]
                                                          (ref ref-obj inlet-idx))))
                                                    inlet))
                                                connections))
                                        inlets)
                          outlets (when (map? event-channel?)
                                    (let [i (apply max (keys event-channel?))]
                                      (map (fn [idx]
                                             (when-let [event-type (get event-channel? idx)]
                                               [(event-channel event-type)]))
                                           (range 0 (inc i)))))
                          obj     (object :object object-name
                                          :args args
                                          :inlets inlets
                                          :outlets outlets)]
                      (-> state
                          (assoc-in [:obj-lookup k] obj)
                          (update :objs conj obj)))))
                {:obj-lookup {}
                 :objs       []}
                objects)]

    (apply object-set (:objs result))))

(defn ->nodes
  "Recursively walks over a collection of nodes (objects or object-sets), and flattens them into a collection of objects"
  [objs]
  (reduce
   (fn [nodes obj]
     (cond
       (and (map? obj) (= ::object-set (:type obj)))
       (into nodes (->nodes (:nodes obj)))

       (and (map? obj) (= ::object (:type obj)))
       (conj nodes obj)))
   []
   objs))

(defn instrument
  "Creates an instrument from an object set.
  The leftmost outlet of the object set will be the inlet for plugout."
  [obj-set]
  {:type    ::instrument
   :id      (genid)
   :nodes   (->nodes [obj-set])
   :plugout [(:outlet obj-set) 0]})

(defn build-router
  [{:keys [router-id]} nodes]
  (let [routes     (->> nodes
                        (reduce
                         (fn [routes {:keys [inlets outlets]}]
                           (into routes
                                 (filter #(= ::route (:type %)))
                                 (mapcat identity (into inlets outlets))))
                         #{})
                        (vec))
        routetable (->> routes
                        (map-indexed (fn [idx route] [(:id route) idx]))
                        (into {}))
        obj        {:object "route"
                    :type   ::object
                    :id     router-id
                    :args   (map :id routes)}]

    [routetable obj]))

(defn build-connection
  [{:keys [router-id]} routetable type id idx connections]
  (map (fn [connection]
         (when connection
           (let [[other-id other-idx] (case (:type connection)
                                        ::route [router-id (get routetable (:id connection))]
                                        ::ref [(:id connection) (:idx connection)]
                                        ::event-channel [(:id connection) 0])]
             (case type
               :input ["script" "connect" other-id other-idx id idx]
               :output ["script" "connect" id idx other-id other-idx]))))
       connections))

(defn build-node-connections
  [env routetable {:keys [inlets outlets id]}]
  (->> (into (map-indexed (partial build-connection env routetable :input id) inlets)
             (map-indexed (partial build-connection env routetable :output id) outlets))
       (mapcat identity)
       (filter identity)))

(defn build-connections
  [env routetable nodes]
  (mapcat (partial build-node-connections env routetable) nodes))

(defn serialize-message-args
  [args]
  args)

(defn build-objects
  [nodes]
  (mapcat (fn [node]
            ;; TODO: something better...
            (let [obj-name (object-name node)]
              (case obj-name
                ;; ugh... this is dirty.
                "message"
                (let [args (serialize-message-args (:args node))]
                  [(into ["script" "send" (:id node) "set"] args)
                   ["script" "nth" (:id node) obj-name 1]
                   (into ["script" "newobject" obj-name])])

                ;; majority of objects can be constructed (as we expect) via these two commands to thispatcher
                [["script" "nth" (:id node) obj-name 1]
                 (into ["script" "newobject" obj-name] (vec (:args node)))])))
          nodes))

(defmulti compile
          "Compiles a jax object into a collection of instructions to be sent to a thispatcher obj"
          (fn [_env obj] (:type obj)))

(defn build-event-channels
  [{:keys [handler-id]} nodes]
  (let [event-channels (->> nodes
                            (mapcat :outlets)
                            (mapcat identity)
                            (filter #(= ::event-channel (:type %))))]
    (mapcat
     (fn [event-channel]
       (let [event-type (-> event-channel :event-type name)
             args       (serialize-message-args [event-type "$1" "$2"])]
         [["script" "newobject" "message"]
          ["script" "nth" (:id event-channel) "message" 1]
          (into ["script" "send" (:id event-channel) "set"] args)
          ["script" "connect" (:id event-channel) 0 handler-id 0]]))
     event-channels)))

(defmethod compile ::instrument
  [{:keys [script-id router-id plugout-id] :as env} {:keys [nodes plugout]}]
  (let [nodes (->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (conj nodes route-obj)]
    (into (conj (build-connections env routetable nodes)
                ["script" "connect" (first plugout) (second plugout) plugout-id 0]
                ["script" "connect" script-id 0 router-id 0])
          (into (build-objects nodes)
                (build-event-channels env nodes)))))

(defmethod compile ::object-set
  [{:keys [script-id router-id] :as env} {:keys [nodes]}]
  (let [nodes (->nodes nodes)
        [routetable route-obj] (build-router env nodes)
        nodes (conj nodes route-obj)]
    (into (conj (build-connections env routetable nodes)
                ["script" "connect" script-id 0 router-id 0])
          (into (build-objects nodes)
                (build-event-channels env nodes)))))

(defn default-env
  [inst]
  (let [inst-id (some-> inst :id name)]
    {:script-id  "mother"
     :midiin-id  "mother-midiin"
     :midiout-id "mother-midiout"
     :plugin-id  "mother-plugin"
     :plugout-id "mother-plugout"
     :patch-dict "mother-dict"
     :handler-id "mother-handler"
     :router-id  (str "mother-router-" inst-id "-" (genid))
     :inst-id    inst-id}))
