(ns dos-installer.installer
  (:require [camel-snake-kebab.core :as csk]
            [clojure.string :as string]
            [dos-installer.utils.deps :as utils.deps]
            [clojure.java.io :as io]
            [clojure.set :as cs]))

(def base-build-iss-filepath "/resources-dos/inno/build.iss")

(defn default-setup-section
  [app-id app-name app-version app-publisher]
  {:section :setup
   :directives {:app-id app-id
                :app-name app-name
                :app-version app-version
                :app-publisher app-publisher
                :default-dir-name (str "{userpf}/" app-name)
                :disable-program-group-page "no"
                :privileges-required "admin"
                :window-visible "yes"
                :uninstallable "yes"
                :output-dir "/work/"
                :output-base-filename (str app-name "-installer")
                :compression "lzma"
                :solid-compression "yes"
                :wizard-style "modern"}})

(def default-language-section
  {:section :languages
   :entries
   [{:name "english" :messages-file "compiler:default:isl"}]})

(def default-tasks-section
  {:section :tasks
   :entries
   [{:name "desktopicon"
     :description "{cm:CreateDesktopIcon}"
     :group-description "{cm:AdditionalIcons}"
     :flags ["unchecked"]}]})

(defn default-files-section
  [app-name jarname]
  {:section :files
   :entries
   [{:source (format "/work/%s.exe" app-name) :dest-dir "{app}" :flags ["ignoreversion"]}
    {:source (format "/work/%s" jarname) :dest-dir "{app}" :flags ["ignoreversion"]}
    {:source "/work/jre/*" :dest-dir "{app}/jre" :flags ["ignoreversion" "recursesubdirs"]}
    {:source "/work/config.json" :dest-dir "{app}" :flags ["ignoreversion"]}
    {:source "/work/resources/*" :dest-dir "{app}/resources" :flags ["ignoreversion" "recursesubdirs" "skipifsourcedoesntexist"]}]})

(defn default-run-section
  [app-name]
  {:section :run
   :entries
   [{:filename (format "{app}/%s.exe" app-name)
     :description (format "{cm:LaunchProgram, {#StringChange (\"%s\", '&', '&&')}}" app-name)
     :flags ["nowait" "hidewizard" "runhidden" "runascurrentuser"]}]})

(defn default-uninstall-section
  [app-name]
  {:section :uninstall-run
   :entries
   [{:filename "{sys}/taskkill.exe" :parameters (format "/f /im %s.exe" app-name) :flags ["skipifdoesntexist" "runhidden"]}]})

(def default-uninstall-delete
  {:section :uninstall-delete
   :entries
   [{:type "filesandordirs" :name "{app}"}]})

(defn default-sections
  [app-id app-name app-version app-publisher jarname]
  {:setup (default-setup-section app-id app-name app-version app-publisher)
   :languages default-tasks-section
   :tasks default-tasks-section
   :files (default-files-section app-name jarname)
   :run (default-run-section app-name)
   :uninstall-delete default-uninstall-delete
   :uninstall-run (default-uninstall-section app-name)})

(defn join-user-sections
  [deps-installer jarname]
  (let [app-name (utils.deps/app-name deps-installer)
        app-id (utils.deps/app-id app-name)
        app-version (utils.deps/app-version deps-installer)
        app-publisher (utils.deps/app-publisher deps-installer)

        us-names (set (map :section (:installer deps-installer)))
        default-sections (default-sections app-id app-name app-version app-publisher jarname)
        df-names (set (keys default-sections))
        diff (cs/difference df-names us-names)]
    (concat (map #(get default-sections %) diff) (:installer deps-installer))))

(defn new-line
  [w]
  (.write w "\n"))

(defn fmt-section
  [section]
  (->> section
       name
       csk/->PascalCase
       (format "[%s]")))

(defn write-section
  [w section]
  (.write w (fmt-section section))
  (new-line w))

(defn ->windows-path
  [s]
  (when s
    (string/replace s #"/" "\\\\")))

(defn fmt-directive
  [k v]
  (str (csk/->PascalCaseString k) "=" (->windows-path v)))

(defn write-directive
  [w directive]
  (doseq [[k v] directive]
    (.write w (fmt-directive k (->windows-path v)))
    (new-line w)))

(defn write-directives
  [w directives]
  (doseq [d (map #(apply hash-map %) directives)]
    (write-directive w d)))

(defn fmt-flags
  [flags]
  (str "Flags: " (string/join #" " flags) " "))

(defn fmt-option
  [k v]
  (let [fmt-value (if (= k :parameters) v (->windows-path v))]
    (format "%s : \"%s\";  " (csk/->PascalCaseString k) fmt-value)))

(defn write-entry
  [entry]
  (reduce
    (fn [acc k]
      (str acc
           (if (= k :flags)
             (fmt-flags (get entry k))
             (fmt-option k (get entry k)))))
    ""
    (keys entry)))

(defn write-entries
  [w entries]
  (when entries
    (doseq [entry entries]
      (.write w (write-entry entry))
      (new-line w))))

(defn build-iss
  [deps user-dir]
  (let [jarname (-> deps utils.deps/executable utils.deps/jarname)
        deps-installer (utils.deps/installer deps)
        filename (str user-dir base-build-iss-filepath)]

    (io/make-parents (io/file filename))

    (doseq [sec (join-user-sections deps-installer jarname)]
      (with-open [w (io/writer filename :append true)]
        (write-section w (:section sec))
        (if (= (:section sec) :setup)
          (write-directives w (:directives sec))
          (write-entries w (:entries sec)))
        (new-line w)))))
