(ns dos-installer.docker
  (:require [clojure.java.io :as io]
            [babashka.process :refer [$ check process]]
            [clojure.string :as string]
            [dos-installer.utils.files :as utils.files]
            [dos-installer.utils.deps :as utils.deps]))

(def base-build-image-name "dos-installer")

(def base-dockerfile-name "Dockerfile")

(defn dockerfile-content
  []
  (-> base-dockerfile-name
      io/resource
      slurp))

(defn print-command
  [p]
  (println "command: " (string/join " " (:cmd p)))
  p)

(defn include-env
  [k v]
  (assoc (into {} (System/getenv)) k v))

(defn build-dockerfile
  [user-dir]
  (utils.files/paste! user-dir base-dockerfile-name (dockerfile-content)))

(defn build-cmd
  [deps]
  (let [jarfile (-> deps utils.deps/executable utils.deps/jarfile)
        jarname (-> deps utils.deps/executable utils.deps/jarname)]
   (-> ^{:inherit true
         :env (assoc (include-env :jarfile jarfile) :jarname jarname)}
       ($ docker build
          -t ~base-build-image-name
          --build-arg ~(str "jarfile=" jarfile)
          --build-arg ~(str "jarname=" jarname)
          -f resources-dos/Dockerfile .)
       print-command
       check)))

(defn get-executable-out
  [deps]
  (let [executable-name (-> deps utils.deps/executable utils.deps/executable-name)]
    (-> ^{:inherit true
          :env (include-env :dos_executable_name executable-name)}
        ($ sh resources-dos/copy-executable.sh)
        print-command
        check)))

(defn get-installer-out
  [deps]
  (let [installer-name (-> deps utils.deps/installer utils.deps/installer-name)]
    (-> ^{:inherit true
          :env (include-env :installer_name installer-name)}
        ($ sh resources-dos/copy-installer.sh)
        print-command
        check)))
