(ns datomock.core
  (:require [datomic.api :as d]
            [datomic.promise]

            [datomock.impl :as impl])
  (:import [java.util.concurrent BlockingQueue LinkedBlockingDeque ExecutionException]
           [datomic Connection Database Log]
           ))

;; ------------------------------------------------------------------------
;; Empty databases

(def empty-db
  "[]
Returns an empty database. Memoized for efficiency."
  (memoize impl/make-empty-db))

;; ------------------------------------------------------------------------
;; Mocked connection

(defrecord MockConnection
  [a_state, forkT, parentLog, a_txq]

  Connection
  (db [_] (:db @a_state))
  (transact [_ tx-data] (doto (datomic.promise/settable-future)
                             (deliver (let [tx-res
                                            (loop []
                                              (let [old-val @a_state
                                                    db (:db old-val)
                                                    tx-res (try (d/with db tx-data)
                                                             (catch Throwable err
                                                               (throw (ExecutionException. err))))
                                                    new-val  (impl/->MockConnState
                                                               (:db-after tx-res)
                                                               (conj (:logVec old-val) (impl/log-item tx-res)))]
                                                (if (compare-and-set! a_state old-val new-val)
                                                  tx-res
                                                  (recur))
                                                ))]
                                        (when-let [^BlockingQueue txq @a_txq]
                                          (.add ^BlockingQueue txq tx-res))
                                        tx-res))
                             ))
  (transactAsync [this tx-data] (.transact this tx-data))

  (requestIndex [_] true)
  (release [_] (do nil))
  (gcStorage [_ olderThan] (do nil))

  (sync [this] (doto (datomic.promise/settable-future)
                 (deliver (.db this))))
  (sync [this t] (.sync this))
  (syncExcise [this t] (.sync this))
  (syncIndex [this t] (.sync this))
  (syncSchema [this t] (.sync this))

  (txReportQueue [_]
    (or @a_txq
        (swap! a_txq #(or % (LinkedBlockingDeque.)))))
  (removeTxReportQueue [_]
    (reset! a_txq nil))

  (log [_] (impl/->ForkedLog parentLog forkT (:logVec @a_state)))
  )

(defn- mock-conn*
  [^Database db, ^Log parent-log]
  (->MockConnection (atom (impl/->MockConnState db [])) (d/next-t db) parent-log (atom nil)))

(defn ^Connection mock-conn
  "Creates a mocked Datomic connection from the given *starting-point* database, or an empty database if none provided.

Implicitly backed by Datomic's speculative writes (#'datomic.api/with).
The returned connection is local and has no URI access.

Exceptions thrown during writes will be systematically wrapped in an j.u.c.ExecutionExecption
to emulate the behaviour of a remote transactor.

The #db(), #transact(), #transactAsync(), #txReportQueue(), #removeTxReportQueue() and #log() methods are fully supported.
Housekeeping methods (#requestIndex(), #release(), #gcStorage()) are implemented as noops.
Sync methods have a trivial implementation.
  "
  ([^Database db] (mock-conn* db nil))
  ([] (mock-conn (empty-db))))

(defn ^Connection fork-conn
  "Forks the given Datomic connection into a local 'mocked' connection.
The starting-point database will be the current database at the time of forking.

Note that if #log() is not supported by `conn` (as is the case for mem connections prior to Datomic 0.9.5407),
the forked connection will support #log(), but the returned Log will only contain transactions committed after forking.

See documentation of #'mock-conn for the characteristics of the returned Connection."
  [^Connection conn]
  (mock-conn* (d/db conn) (d/log conn)))




