(ns zookareg.core
  "Utilities for configuring, starting and halting Zookareg.

  A running Zookareg instance consists of:
  - a Kafka Broker
  - a Curator (Zookeeper) TestingServer
  - a Confluent Schema Registry"
  (:require [clojure.tools.logging :as log]
            [integrant.core :as ig]
            [zookareg.kafka :as ka]
            [zookareg.schema-registry :as reg]
            [zookareg.state :as state]
            [zookareg.utils :as ut]
            [zookareg.zookeeper :as zoo]))

(defn ->config
  "Creates a config from ports and, optionallly, Kafka and Schema Registry configuration overrides."
  ([ports]
   (->config ports nil nil))
  ([ports kafka-config-overrides]
   (->config ports kafka-config-overrides nil))
  ([ports kafka-config-overrides schema-registry-config-overrides]
   {:ports                  ports
    :kafka-config           (ka/->config ports kafka-config-overrides)
    :schema-registry-config (reg/->config ports schema-registry-config-overrides)}))

(def default-ports
  {:kafka           9092
   :zookeeper       2181
   :schema-registry 8081})

(def default-config (->config default-ports))

(defn ->ports
  "Util for generating a ::ports map."
  [kafka zookeeper schema-registry]
  {:kafka           kafka
   :zookeeper       zookeeper
   :schema-registry schema-registry})

(defn ->available-ports
  "Returns a ::ports map with currently available ports."
  []
  (->ports (ut/->available-port)
           (ut/->available-port)
           (ut/->available-port)))

(defn ->available-ports-config
  "Returns a ::config map with currently available ::ports."
  []
  (->config (->available-ports)))

(defn- ->ig-config
  "Returns an Integrant system config form a ::config. Implementation detail."
  [config]
  {:zookareg.schema-registry/schema-registry
   {:ports      (:ports config)
    :config     (:schema-registry-config config)
    :_kafka     (ig/ref :zookareg.kafka/kafka)
    :_zookeeper (ig/ref :zookareg.zookeeper/zookeeper)}

   :zookareg.kafka/kafka
   {:ports      (:ports config)
    :config     (:kafka-config config)
    :_zookeeper (ig/ref :zookareg.zookeeper/zookeeper)}

   :zookareg.zookeeper/zookeeper
   {:ports (:ports config)}})

(defn halt-zookareg!
  "Halts the running Zookareg instance."
  []
  (when @state/state
    (swap! state/state
           (fn [s]
             (ig/halt! (:system s))
             nil))))

(defn init-zookareg
  "Starts a Zookareg instance."
  ([] (init-zookareg default-config))
  ([config]
   (let [ig-config (->ig-config config)]
     (log/info "starting ZooKaReg with config:" config)
     (try
       (halt-zookareg!)
       (ig/load-namespaces ig-config)
       ;; TODO stick in the same atom!
       (reset! state/state
               {:system (ig/init ig-config)
                :config ig-config})
       (catch clojure.lang.ExceptionInfo ex
         ;; NOTE tears down partially initialised system
         (ig/halt! (:system (ex-data ex)))
         (throw (.getCause ex)))))))

(defn with-zookareg-fn
  "Starts up zookareg with the specified configuration; executes the function then shuts down."
  ([config f]
   {:pre [(map? config) (fn? f)]}
   (try
     (init-zookareg config)
     (f)
     (finally
       (halt-zookareg!))))
  ([f]
   (with-zookareg-fn default-config f)))

(defmacro with-zookareg
  "Starts up zookareg with the specified configuration; executes the body then shuts down."
  [config & body]
  `(with-zookareg-fn ~config (fn [] ~@body)))
