(ns zookareg.schema-registry
  "Utilities for configuring, starting and halting an embedded Confluent Schema
  Registry."
  (:require [integrant.core :as ig]
            [me.raynes.fs :as fs]
            [zookareg.utils :as ut])
  (:import [io.confluent.kafka.schemaregistry.rest SchemaRegistryConfig SchemaRegistryRestApplication]
           java.io.File
           org.eclipse.jetty.server.Server))

(defn ->config
  "Creates a Confluent Schema Registry config from ports and optionally overrides."
  ([ports] (->config ports nil))
  ([{:keys [zookeeper kafka schema-registry] :as _ports} overrides]
   (merge {:listeners                           (str "http://0.0.0.0:" schema-registry)
           :kafkastore.connection.url           (str "localhost:" zookeeper)
           :kafkastore.bootstrap.servers        (str "PLAINTEXT://localhost:" kafka)
           :kafkastore.topic.replication.factor 1}
          overrides)))

(defn- ^File ->config-file [config]
  (let [props (ut/m->properties config)
        file  (fs/temp-file "schema-registry" ".properties")]
    (ut/store-properties props file)
    file))

(defn ->schema-registry
  "Starts a Confluent Schema Registry server from a config produced with `->config`."
  [config]
  (let [config (SchemaRegistryConfig. (.getAbsolutePath (->config-file config)))
        app    (SchemaRegistryRestApplication. ^SchemaRegistryConfig config)
        ^Server server (.createServer app)]
    (.start server)
    server))

(defn halt!
  "Stops a Confluent Schema Registry server."
  [^Server server]
  (when server
    (.stop server)))

(defmethod ig/init-key ::schema-registry [_ {:keys [config]}]
  (->schema-registry config))

(defmethod ig/halt-key! ::schema-registry [_ server]
  (halt! server))
