(ns scribe.avro.schema-specs
  "Validations for Avro schemas according to
  https://avro.apache.org/docs/current/spec.html"
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as string]))

(defn- name+ [x] (when x (name x)))

(defn- type= [t]
  (fn [x]
    (= t (-> x :type name+))))

(defn- x-name
  [x]
  (try
    (name x)
    (catch Exception _
      :clojure.spec.alpha/invalid)))

;;,------------------------------------------------------------
;;| Names: https://avro.apache.org/docs/current/spec.html#names
;;`------------------------------------------------------------
(defn valid-name?
  "Returns whether or not a given string is a valid Avro name.

  See: https://avro.apache.org/docs/current/spec.html#names"
  [s]
  (boolean (and (re-matches #"^[A-Za-z_].*$" s)
                (re-matches #"^.[A-Za-z0-9_]*$" s))))

(defn valid-namespace?
  "Returns whether or not a given string is a valid Avro namespace.

   See: https://avro.apache.org/docs/current/spec.html#names"
  [x]
  (every? valid-name? (string/split x #"\.")))

(s/def :avro/name
  (s/and (s/conformer x-name)
         valid-name?))
(s/def :avro/namespace
  (s/and (s/conformer x-name)
         valid-namespace?))
;; NOTE A bit hacky, but does the job for now
(s/def :avro/qualified-name :avro/namespace)

(s/def :avro.name.wrapped/type :avro/name) ;; oh ffs..
(s/def :avro.name/wrapped
  (s/and (s/keys :req-un [:avro.name.wrapped/type])
         (comp not (type= "record"))))

;;,----------------------------------------------------------------
;;| Primitive Types:
;;| https://avro.apache.org/docs/current/spec.html#schema_primitive
;;`----------------------------------------------------------------
(defn primitive-type?
  "Returns whether or not a given string/keyword is an Avro Primitive Type.

   See: https://avro.apache.org/docs/current/spec.html#schema_primitive"
  [x]
  (#{"null"
     "int" "long" "float" "double"
     "bytes" "boolean" "string"} (name x)))
(s/def :avro/primitive
  (s/and :avro/name primitive-type?))

;;,---------------
;;| Logical Types:
;;| https://avro.apache.org/docs/current/spec.html#Logical+Types
;;`---------------
(defn logical-type?
  "Returns whether or not a given string/keyword is an Avro Logical Type.

   See: https://avro.apache.org/docs/current/spec.html#Logical+Types"
  [x]
  (#{"decimal" "date" "time-millis" "time-micros" "timestamp-millis" "timestamp-micros"}  (name x)))

;; TODO finish implementing these

;;,--------------------------------------------------------------
;;| Arrays: https://avro.apache.org/docs/current/spec.html#Arrays
;;`--------------------------------------------------------------
(s/def :avro.array/items
  ;; NOTE s/and = hack for yet undefined spec
  (s/and :avro/schema))
(s/def :avro/array
  (s/and (type= "array")
         (s/keys :req-un [:avro.array/items])))

;;,----------------------------------------------------------------------
;;| Records: https://avro.apache.org/docs/current/spec.html#schema_record
;;`----------------------------------------------------------------------
(s/def :avro.record.field/type
  ;; NOTE s/and = hack for yet undefined spec
  (s/and :avro/schema))
(s/def :avro.record/field
  (s/and (s/keys :req-un [:avro/name
                          :avro.record.field/type]
                 :opt-un [::namespace])))
(s/def :avro.record/fields
  (s/and (s/coll-of :avro.record/field)
         seq))
(s/def :avro/record
  (s/and (type= "record")
         (s/keys :req-un [:avro/name
                          :avro.record/fields]
                 :opt-un [:avro/namespace])))

;;,--------------------------------------------------------------
;;| Unions: https://avro.apache.org/docs/current/spec.html#Unions
;;`--------------------------------------------------------------
(s/def :avro/union
  (s/and sequential?
         (s/coll-of :avro/schema)
         seq))

;; TODO plug in abracad/parse-schema for final sanity check

;;,-----------------------------------------------------------------
;;| The Avro Schema in all its Glory. Praise be to the Lord of Types
;;`-----------------------------------------------------------------
(s/def :avro/schema
  ;; NOTE order is important here!
  (s/or :primitive :avro/primitive
        ;; the `type-name` here refers to a derived type name (since
        ;; :avro/primitive would have already been matched). A Record name, for
        ;; example. https://avro.apache.org/docs/current/spec.html#schemas
        :type-name :avro/name
        :qualified-type-name :avro/qualified-name
        :record :avro/record
        ;; TODO enums
        :array :avro/array
        ;; TODO maps
        ;; TODO fixed
        :union :avro/union
        :gratuitious-map-type-wrapper :avro.name/wrapped))
(s/def ::avro-schema :avro/schema)
