(ns scribe.decimal
  (:require [abracad.avro :as aa]
            [chronos.utils :as u]
            [clojure.java.io :as io]
            [clojure.spec.alpha :as s])
  (:import java.nio.ByteBuffer
           [org.apache.avro Conversions$DecimalConversion LogicalTypes]))

(defn set-scale [^BigDecimal bd scale] (.setScale bd scale))

(defn set-precision [^BigDecimal bd precision]
  (let [prec (+ (.scale bd)
                (- precision (.precision bd)))]
    (.setScale bd prec)))

(defn set-precision-scale [bd precision scale]
  (-> bd
      (set-precision precision)
      (set-scale scale)))

(defmacro ->decimal-conformer [& {:keys [precision scale]}]
  `(s/conformer
    (fn  [bd#]
     ;; FIXME ??? TODO is there any saner way?
      {:scribe.conformer.fn/name ::->decimal-conformer
       :precision                ~precision
       :scale                    ~scale}
      (if-not (decimal? bd#)
        :clojure.spec.alpha/invalid
        (try
          (set-precision-scale bd# ~precision ~scale)
          (catch ArithmeticException _#
            :clojure.spec.alpha/invalid))))))

(defn- ->LogicalType [precision scale]
  (LogicalTypes/decimal (int precision) (int scale)))

(defn- ->avro-schema [type precision scale]
  (aa/parse-schema
   {:type        type
    :logicalType :decimal
    :precision   precision
    :scale       scale}))

(defn decimal->bytes [bd precision scale]
  (.toBytes (Conversions$DecimalConversion.)
            (set-precision-scale bd precision scale)
            (->avro-schema :bytes precision scale)
            (->LogicalType precision scale)))

(defn decimal->fixed [bd precision scale]
  ;; TODO FIXME not really implemented
  (u/not-implemented!)
  (.toFixed (Conversions$DecimalConversion.)
            (set-scale bd scale)
            (->avro-schema :fixed precision scale)
            (->LogicalType precision scale)))

(defn byte-buffer?
  [x]
  (isa? (class x) java.nio.ByteBuffer))

(defn- ->buf [buf-or-bytes]
  (if (byte-buffer? buf-or-bytes)
    buf-or-bytes
    (ByteBuffer/wrap buf-or-bytes)))

(defn bytes->decimal [buffer-or-bytes precision scale]
  (.fromBytes (Conversions$DecimalConversion.)
              (->buf buffer-or-bytes)
              (->avro-schema :bytes precision scale)
              (->LogicalType precision scale)))

(defn fixed->decimal [fixed precision scale]
  ;; TODO FIXME not really implemented
  (u/not-implemented!)
  (.fromFixed (Conversions$DecimalConversion.)
              fixed
              (->avro-schema :fixed precision scale)
              (->LogicalType precision scale)))
