(ns scribe.avro.core
  "Scribe spec to Avro schema conversions, and Avro
  pre-serialization/post-deserialization."
  (:require [abracad.avro :as avro]
            [clojure.spec.alpha :as s]
            [medley.core :refer [assoc-some]]
            [scribe.avro.serde-utils :as serde]
            [scribe.utils :refer :all]))

(def avro-decimal-precision 32)
(def avro-decimal-scale 32)

;;,-----------------------------------------------
;;| https://avro.apache.org/docs/current/spec.html
;;`-----------------------------------------------

(defn- avro-enum?
  "non-empty, all keyword sets are avro enums"
  [x]
  ((every-pred set?
               not-empty
               (partial every? keyword?)) x))

(defn- spec-variant [spec-name]
  (cond
    (symbol? spec-name) spec-name
    :else
    (let [form (s/form spec-name)
          _    (type form)]
      (cond
        (avro-enum? form) :scribe.avro/enum
        (symbol? form)    form
        (seq? form)       (first form)))))

(defn- ->avro-n+ns [spec-name]
  (-> {:name (name spec-name)}
      (assoc-some :namespace (namespace spec-name))))

(defn- ->avro-ref [spec-name]
  (-> (->avro-n+ns spec-name)
      (qualify-keys :avro)
      (with-meta {:spec-variant (spec-variant spec-name)})))

(defn- assoc-avro-ref
  "To be called after defining an avro type so that subsequent calls will just
  use its fully qualified name"
  [avro-refs spec-name]
  (assoc avro-refs spec-name (->avro-ref spec-name)))

;;,------------------------
;;| Avro Schema conversions
;;`------------------------
(defmulti ^:private ->avro-schema-
  "Returns a tuple: avro-refs+schema. The avro-refs is updated so that schemas are
  only defined on the first instance and then referred to by fully qualified
  name."
  (fn [avro-refs spec-name]
    (if (get avro-refs spec-name)
      :scribe.avro/ref
      (spec-variant spec-name))))

(defmethod ->avro-schema- :scribe.avro/ref
  [avro-refs spec-name]
  (let [s  (get avro-refs spec-name)
        n  (:avro/name s)
        ns (:avro/namespace s)]
    [avro-refs (if ns (str ns "." n) n)]))

(defmethod ->avro-schema- 'clojure.core/string?
  [avro-refs _]
  [avro-refs {:type :string}])

(defmethod ->avro-schema- 'clojure.core/boolean?
  [avro-refs _]
  [avro-refs {:type :boolean}])

(defmethod ->avro-schema- 'clojure.core/decimal?
  [avro-refs spec-name]
  ;; TODO ??? how do we treat custom precision/scale?
  [avro-refs {:type :bytes
              :logicalType    :decimal
              :precision      avro-decimal-precision
              :scale          avro-decimal-scale}])

(defmethod ->avro-schema- :scribe/date
  [avro-refs spec-name]
  (let [s (get avro-refs spec-name)]
    [avro-refs
     {:type        :int
      :logicalType :date}]))

(defmethod ->avro-schema- :scribe/time-millis
  [avro-refs spec-name]
  (let [s (get avro-refs spec-name)]
    [avro-refs
     {:type        :int
      :logicalType :time-millis}]))

(defmethod ->avro-schema- :scribe/time-micros
  [avro-refs spec-name]
  (let [s (get avro-refs spec-name)]
    [avro-refs
     {:type        :long
      :logicalType :time-micros}]))

(defmethod ->avro-schema- :scribe/timestamp-millis
  [avro-refs spec-name]
  (let [s (get avro-refs spec-name)]
    [avro-refs
     {:type        :int
      :logicalType :timestamp-millis}]))

(defmethod ->avro-schema- :scribe/timestamp-micros
  [avro-refs spec-name]
  (let [s (get avro-refs spec-name)]
    [avro-refs
     {:type        :long
      :logicalType :timestamp-micros}]))

(defmethod ->avro-schema- :scribe.avro/enum
  [avro-refs spec-name]
  [(assoc-avro-ref avro-refs spec-name)
   (merge (->avro-n+ns spec-name)
          {:symbols (s/form spec-name)
           :type    :enum})])

(defmethod ->avro-schema- 'clojure.spec.alpha/coll-of
  [avro-refs spec-name]
  (let [form (s/form spec-name)
        ;; TODO support opts
        pred (second form)
        [avro-refs
         items-schema] (->avro-schema- avro-refs pred)
        schema         (merge (->avro-n+ns spec-name)
                              {:items items-schema
                               :type  :array})]
    [avro-refs schema]))

(defn- ->record-field
  [avro-refs spec-name]
  (let [[avro-refs schema] (->avro-schema- avro-refs spec-name)]
    [avro-refs {:name (name spec-name)
                :type schema}]))
(defmethod ->avro-schema- 'clojure.spec.alpha/keys
  [avro-refs spec-name]
  (let [form   (s/form spec-name)
        opts   (kwargs->m (rest form))
        ;; TODO req, opt-un, opt
        req-un (:req-un opts)
        [avro-refs fields]
        (reduce (fn [[avro-refs field-schemas] spec-name]
                  (let [[avro-refs field-schema] (->record-field avro-refs spec-name)]
                    [avro-refs (conj field-schemas field-schema)]))
                [avro-refs []]
                req-un)]
    [(assoc-avro-ref avro-refs spec-name)
     (merge (->avro-n+ns spec-name)
            {:fields fields
             :type   :record})]))

(defn- or-preds
  [spec-name]
  (-> spec-name s/form rest kwargs->m vals))

(defmethod ->avro-schema- 'clojure.spec.alpha/or
  [avro-refs spec-name]
  (let [form (s/form spec-name)
        preds (-> (rest form) kwargs->m vals)]
    (->> preds
         (reduce (fn [[avro-refs members-schemas] spec-name]
                   (let [[avro-refs member-schema]
                         (->avro-schema- avro-refs spec-name)]
                     [avro-refs (conj members-schemas member-schema)]))
                 [avro-refs []]))))

(defn ->avro-schema
  "Generates an Avro schema for spec-name.

  spec-name: the key of the spec to convert"
  [spec-name]
  (second (->avro-schema- {} spec-name)))

;;,-----------------------
;;| Avro pre-serialization
;;`-----------------------
(declare pre-serialize)

(defmulti ^:private pre-serialize- (fn [spec-name _v] (spec-variant spec-name)))

(defmethod pre-serialize- 'clojure.core/decimal?
  [spec-name v]
  ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
  (serde/big-decimal->unscaled-int-bytes avro-decimal-scale
                                         v))

(defmethod pre-serialize- 'clojure.spec.alpha/coll-of
  [spec-name v]
  (let [pred (-> spec-name s/form second)]
    (map (partial pre-serialize pred) v)))

(defmethod pre-serialize- 'clojure.spec.alpha/keys
  [spec-name v]
  (let [form        (s/form spec-name)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO req, opt, opt-un
    ;; TODO simplify with map-kv
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial pre-serialize k-spec-name)))
            v
            req-un-keys)))

(defn- or-pred-for [or-spec-name v]
  "Given an s/or spec, and a value v, it returns the pred for v"
  (let [form       (s/form or-spec-name)
        matching-k (first (s/conform or-spec-name v))
        k->pred    (kwargs->m (rest form))]
    (k->pred matching-k)))
(defmethod pre-serialize- 'clojure.spec.alpha/or
  [spec-name v]
  (pre-serialize (or-pred-for spec-name v) v))

(defmethod pre-serialize- :default [_ v] v)

(defn pre-serialize
  "Prepares a value for encoding to Avro.

  For example, it converts `BigDecimal`s to bytes for `:decimal`s,
  `:timestamp-millis` to integers and so on, according to the Avro
  specification."
  [spec-name v]
  (pre-serialize- spec-name v))

;;,------------------------
;;| Avro post-deserialization
;;`------------------------
(declare post-deserialize)

(defmulti ^:private post-deserialize- (fn [spec-name _v] (spec-variant spec-name)))

(defmethod post-deserialize- 'clojure.core/decimal?
  [spec-name v]
  ;; TODO ??? soo.. precision? check it in the scribe.spec.core?
  (serde/unscaled-int-bytes->big-decimal avro-decimal-scale v))

(defmethod post-deserialize- :scribe.avro/enum
  [_ v]
  (keyword v))

(defmethod post-deserialize- 'clojure.spec.alpha/keys
  [spec-name v]
  (let [form        (s/form spec-name)
        opts        (kwargs->m (rest form))
        req-un-keys (:req-un opts)]
    ;; TODO req, opt, opt-un
    ;; TODO simplify with map-kv
    (reduce (fn [m k-spec-name]
              (update m
                      (disqualify-key k-spec-name)
                      (partial post-deserialize k-spec-name)))
            v
            req-un-keys)))

(defmethod post-deserialize- 'clojure.spec.alpha/coll-of
  [spec-name v]
  (let [pred (-> spec-name s/form second)]
    (map (partial post-deserialize pred) v)))

(defn- or-avro-value-spec-name
  "Given the preds from an s/or form, it returns the spec-name of the value"
  [or-preds v]
  (let [v-type (-> v type keyword)]
    (->> or-preds
         (filter (partial = v-type))
         first)))
(defmethod post-deserialize- 'clojure.spec.alpha/or
  [spec-name v]
  (if-let [sn (or-avro-value-spec-name (or-preds spec-name) v)]
    (post-deserialize sn v)
    v))

(defmethod post-deserialize- :default [_ v] v)

(defn post-deserialize
  "Processes a deserailized value from Avro.

  For example, it converts bytes to a BigDecimal for `:decimal`s, integers to
  Instants for `:timestamp-millis` and so on, according to the Avro
  specification."
  [spec-name v]
  (post-deserialize- spec-name v))

;;,------
;;| SerDe
;;`------

(defn binary-encoded [spec-name & records]
  (let [s (->avro-schema spec-name)
        rs (map (partial pre-serialize spec-name) records)]
    (apply avro/binary-encoded s rs)))

(defn decode [spec-name source]
  (let [s (->avro-schema spec-name)]
    (post-deserialize spec-name (avro/decode s source))))

;; TODO all all the abracads
