(ns parseq.combinators
  "Combinators for `parseq.parsers` and similar."
  (:refer-clojure :exclude [or and merge peek])
  (:require [parseq.utils :as pu]))

(defn return
  "A parser that does nothing and always succeeds. It returns the input
  unchanged and the supplied `v` as a result."
  [v]
  (fn [input] [v input]))

(defn bind
  "This is a monadic bind. A.k.a. `>>=`.

  Its type is:

  bind :: [Parser a, (a -> Parser b)] -> Parser b"
  [p f]
  (fn [input]
    (pu/match-parse (pu/parse p input)
                    [r rsin] (pu/parse (f r) rsin)
                    f f)))

(defn fmap
  "Applies function `f` to the result of `p`.

  Its type is:

  fmap :: [Parser a, (a -> b)] -> Parser b"
  [f p]
  (bind p #(return (f %))))

(defn or
  "Tries `parsers` one after another. Returns the result from the first parser
  that succeeds. If all parsers fail, it fails."
  [& parsers]
  (fn [input]
    (loop [parsers  parsers
           failures []]
      (if-let [[p & ps] (not-empty parsers)]
        (pu/match-parse (pu/parse p input)
                        [r rsin] [r rsin]
                        f (recur ps (conj failures f)))
        (pu/->failure "or-c had no more parsers"
                      {:failures failures})))))

(defn and
  "Applies `parsers` one after another. If any parser fails, it fails."
  [& parsers]
  (fn [input]
    (loop [parsers parsers
           res     []
           input   input]
      (if-let [[p & ps] (not-empty parsers)]
        (pu/match-parse (pu/parse p input)
                        [r rsin] (recur ps (conj res r) rsin)
                        f f)
        [res input]))))

(defn one?
  "Optionally parses one `p`, returning a sequential coll containing it, if
  found. If `p` doesn't match, returns an empty coll."
  [p]
  (fn [input]
    (pu/match-parse (pu/parse p input)
                    [r rsin] [[r] rsin]
                    _f [[] input])))
(def ^{:doc "Alias to `one?`"} optional one?)

(defn many*
  "Parse `p` 0 or more times. Similar to `*` in regular expressions."
  [p]
  (fn [input]
    (loop [results    []
           rest-input input]
      (pu/match-parse (pu/parse p rest-input)
                      [r rsin] (recur (conj results r) rsin)
                      _ [results rest-input]))))

(defn many+
  "Parse `p` 1 or more times. Similar to `+` in regular expressions."
  [p]
  (bind p
        (fn [r]
          (fmap #(cons r %)
                (many* p)))))

(defn merge
  "Applies `parsers` in order and then merges their results into one big fat
  map."
  [parsers]
  (fmap (partial apply clojure.core/merge)
        (apply and parsers)))

(defn peek
  "Peeks with `p` (and fails if `p` fails). Does not consume any input."
  [p]
  (fn [input]
    (pu/match-parse (pu/parse p input)
                    [r _rsin] [r input]
                    f f)))

(defn skip*
  "Skips 0 or more `p`."
  [p]
  (fn [input]
    (loop [rest-input input]
      (pu/match-parse (pu/parse p rest-input)
                      [_ rsin] (recur rsin)
                      _ [nil rest-input]))))

(defn skip+
  "Skips 1 or more `p`."
  [p]
  (bind p (fn [_] (skip* p))))
