(ns parseq.combinators
  "Combinators for `parseq.parsers` and similar"
  (:refer-clojure :exclude [or and merge peek])
  (:require [parseq.utils :as pu]))

(defmacro match-parse
  [& body]
  (let [[parse-body
         success-binds success-body
         failure-binds fail-body] body]
    `(let [result# ~parse-body]
       (if (pu/success? result#)
         (let [~success-binds result#]
           ~success-body)
         (let [~failure-binds result#]
           ~fail-body)))))

(defmacro return
  "A parser that does nothing and always succeeds. It returns the input
  unchanged and the supplied `v` as a result."
  [v]
  `(fn [input#] [~v input#]))

(defmacro bind
  "This is a monadic bind. A.k.a. >>=

  Its type is:

  bind :: [Parser a, (a -> Parser b)] -> Parser b"
  [p f]
  `(fn [input#]
     (match-parse (pu/parse ~p input#)
                  [r# rsin#] (pu/parse (~f r#) rsin#)
                  fail# fail#)))

(defmacro fmap
  "Applies function `f` to the result of `p`

  Its type is:

  fmap :: [Parser a, (a -> b)] -> Parser b"
  [f p]
  `(bind ~p (fn [x#] (return (~f x#)))))

(defmacro or
  "Tries `parsers` in sequence and returns the first one that succeeds. If they
  all fail, it fails."
  [& parsers]
  `(fn [input#]
     (loop [parsers#  [~@parsers]
            failures# []]
       (if-let [[p# & ps#] (not-empty parsers#)]
         (match-parse (pu/parse p# input#)
                      [r# rsin#] [r# rsin#]
                      fail# (recur ps# (conj failures# fail#)))
         (pu/->failure "or-c had no more parsers"
                       {:parsers-failures failures#})))))

(defmacro and
  [& parsers]
  `(fn [input#]
     (loop [parsers#    [~@parsers]
            res#        []
            rest-input# input#]
       (if-let [[p# & ps#] (not-empty parsers#)]
         (match-parse (pu/parse p# rest-input#)
                      [r# rsin#] (recur ps# (conj res# r#) rsin#)
                      fail# fail#)
         [res# rest-input#]))))

(defmacro one?
  "Optionally parses one `p`, returning a sequential coll containing it if found. If `p`
  doesn't match, returns an empty coll."
  [p]
  `(fn [input#]
     (match-parse (pu/parse ~p input#)
                  [r# rsin#] [[r#] rsin#]
                  _f# [[] input#])))

(defmacro many*
  "Parse `p` 0 or more times. Similar to `*` in regular expressions."
  [p]
  `(fn [input#]
     (loop [results#    []
            rest-input# input#]
       (match-parse (pu/parse ~p rest-input#)
                    [r# rsin#] (recur (conj results# r#) rsin#)
                    _# [results# rest-input#]))))

(defmacro many+
  "Parse `p` 1 or more times. Similar to `+` in regular expressions."
  [p]
  `(bind ~p
         (fn [r#]
           (fmap (fn [x#] (cons r# x#))
                 (many* ~p)))))

(defmacro merge
  "Applies `parsers` in order and then merges their results into one big fat
  map."
  [parsers]
  `(fn [input#]
     (loop [parsers#   [~@parsers]
            rest-input# input#
            results#    {}]
       (if-let [[p# & ps#] (not-empty parsers#)]
         (match-parse (pu/parse p# rest-input#)
                      [r# rsin#] (recur ps# rsin# (conj results# r#))
                      fail# fail#)
         [results# rest-input#]))))

(defmacro peek
  "Peeks with p (and fails if p fails). Does not consume any input"
  [p]
  `(fn [input#]
     (match-parse (pu/parse ~p input#)
                  [r# _rsin#] [r# input#]
                  fail# fail#)))

(defmacro skip*
  "Skips 0 or more p."
  [p]
  `(fmap (fn [_#] nil) (many* ~p)))

(defmacro skip+
  "Skips 1 or more p."
  [p]
  `(fmap (fn [_#] nil) (many+ ~p)))
