(ns dbwalk.schema-detect
  "Simple database schema detection using information schema. Tested with postgres."
  (:require [clojure.java.jdbc :as jdbc]))


(def ^:private foreign-key-query
  "SELECT DISTINCT
   tc.table_name, kcu.column_name, ccu.table_name AS foreign_table_name, ccu.column_name AS foreign_column_name
   FROM information_schema.table_constraints AS tc
   JOIN information_schema.key_column_usage AS kcu ON tc.constraint_name = kcu.constraint_name
   JOIN information_schema.constraint_column_usage AS ccu ON ccu.constraint_name = tc.constraint_name
   WHERE constraint_type = 'FOREIGN KEY'")

(defn- get-information-schema-foreign-keys [db-spec]
  (->> (jdbc/query db-spec foreign-key-query)
       (map #(hash-map (keyword (:table_name %))
                       [(hash-map :type :many-to-one
                                  :target (keyword (:foreign_table_name %))
                                  :column-name (keyword (:column_name %)))]))
       (apply merge-with concat)))

(def ^:private primary-key-query
  "SELECT tc.table_name, kc.column_name
   FROM information_schema.table_constraints tc, information_schema.key_column_usage kc
   WHERE tc.constraint_type = 'PRIMARY KEY'
        AND kc.table_name = tc.table_name
        AND kc.table_schema = tc.table_schema
        AND kc.constraint_name = tc.constraint_name")

(defn- get-information-schema-primary-keys "Creates a flat tablename -> primary key name -mapping"
  [db-spec]
  (->> (jdbc/query db-spec primary-key-query)
       (map #(hash-map (keyword (:table_name %)) (keyword (:column_name %))))
       (apply merge)))


(defmulti detect-schema-for-database "Attempts to autodetect many-to-one relations and primary keys."
          (fn [type db-spec]
            type))

;; Tested on postgresql-9.4 and 9.5
(defmethod detect-schema-for-database :default [type db-spec]
  (let [primary-keys (get-information-schema-primary-keys db-spec)
        foreign-keys (get-information-schema-foreign-keys db-spec)
        table-names  (apply hash-set (concat (keys primary-keys) (keys foreign-keys)))]
    (for [table table-names]
      (let [pk (get primary-keys table)
            fk (get foreign-keys table [])]
        (hash-map :name table
                  :primary-key pk
                  :dbwalk/relations fk)))))

(defn detect-schema "Attempts to autodetect many-to-one relations and primary keys."
  (
   [db-spec]
   (detect-schema db-spec :default))
  (
   [db-spec type]
   (detect-schema-for-database type db-spec)))


