(ns dbwalk.output.map
  "The default output, where everything that was read from the DB is returned as a nested map."
  (:require [dbwalk.entity-node :as en]
            [dbwalk.graph :as graph]
            [dbwalk.relations :as relation]))


(defn- get-relation-name [graph from to]
  (relation/created-attribute-name (graph/relation-for-link graph from to)))

(defn data-with-table-namespace
  "Adds table-name as the ns for all keys in node."
  [node table-name]
  (zipmap (map #(keyword (name table-name) (name %)) (keys node)) (vals node)))

(declare hierarchy-dump-graph-from-node)

(defn- dump-child [graph transform child]
  (hierarchy-dump-graph-from-node graph transform child))


(defn- group-children-by-relation [graph start-node transform children]
  (map (fn [child]
         (let [r     (graph/relation-for-link graph start-node child)
               child (dump-child graph transform child)]
           (hash-map
             (relation/created-attribute-name r)
             (if (relation/multiple-targets? r)
               [child]
               child))))
       children))

(defn- merge-child-vectors [a b]
  (assert (vector? a))
  (assert (vector? b))
  (into a b))

(defn- merge-grouped-children [table-to-child-list]
  (apply merge-with merge-child-vectors table-to-child-list))

(defn- hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [graph transform start-node]
  (if-let [children (seq (loom.graph/successors graph start-node))]
    (->> children
         (graph/to-insertion-order graph)
         (group-children-by-relation graph start-node transform)
         (merge-grouped-children)
         (merge (transform (en/data start-node) (graph/table-for-entity graph start-node))))
    (transform (en/data start-node) (graph/table-for-entity graph start-node))))

(defn- to-map-with-transform
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :dbwalk/eager query. }"
  [graph transform]
  (en/data-only
    (->> (graph/find-root-items graph)
         (graph/to-insertion-order graph)
         (map (partial hierarchy-dump-graph-from-node graph transform))
         (vec))))

(defn graph->map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :dbwalk/eager query. }
  The keys for each entity map will not a namespace."
  [graph]
  (to-map-with-transform graph (fn [a b] a)))

(defn graph->namespaced-map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :dbwalk/eager query. }
  The keys for each entity map will have the entity's table as their namespace."
  [graph]
  (to-map-with-transform graph data-with-table-namespace))
