(ns dbwalk.output.filtered
  "Implements the 'filtered' output function, which allows selecting only required attributes and tables."
  (:require [dbwalk.graph :as graph]
            [schema.core :as s]
            [dbwalk.entity-node :as en]
            [dbwalk.output.map :as map]
            [loom.graph :as lg]
            [dbwalk.input.filter-data :refer :all]))

(s/defn ^:private hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [filter-data :- FilterData
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode])

(s/defn merge-children
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode])


(s/defn ^:private keep-or-ignore-node
  "If this node is to be kept, returns {tablename -> node}.
   Otherwise, returns this node's children."
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   node :- en/EntityNode]
  (let [tables-to-keep (apply hash-set (keys filter-data))]
    (if (contains? tables-to-keep (graph/table-for-entity graph node))
      (hash-map (graph/table-for-entity graph node)
                [(hierarchy-dump-graph-from-node filter-data graph node)])
      (merge-children filter-data graph node))))

(defn- to-insertion-order [graph children]
  (graph/to-insertion-order graph children))

(s/defn ^:private merge-children
  "Sorts and filters this node's children."
  [filter-data :- s/Any
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode]
  (if-let [children (seq (loom.graph/successors graph start-node))]
    (->> children
         (to-insertion-order graph)
         (map #(keep-or-ignore-node filter-data graph %))
         (apply merge-with #(into (vec %1) %2)))))

(defn- select-columns [e cols]
  (if (contains? cols :*)
    e
    (select-keys e cols)))

(s/defn ^:private hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [filter-data :- FilterData
   graph :- (s/protocol lg/Digraph)
   start-node :- en/EntityNode]
  (let [this-node-table        (graph/table-for-entity graph start-node)
        columns-for-this-table (get filter-data this-node-table)
        filtered-start-node    (select-columns (en/data-only start-node) columns-for-this-table)]
    (->> (merge-children filter-data graph start-node)
         (en/data-only)
         (merge (map/data-with-table-namespace filtered-start-node (name this-node-table))))))


(s/defn graph->filtered-map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :dbwalk/eager query. }"
  [columns :- [s/Keyword]
   graph :- (s/protocol lg/Digraph)]
  (let [filter-data (columns-to-filter-data columns)]
    (->> graph
         (graph/find-root-items)
         (graph/to-insertion-order graph)
         (map #(keep-or-ignore-node filter-data graph %))
         (apply merge-with #(into (vec %1) %2)))))



