(ns dbwalk.graph
  "Encapsulates loom.graph."
  (:require [dbwalk.relations :as relation]
            [loom.graph :as lg]
            [loom.attr :as la]
            [dbwalk.entity-node :as en]
            [dbwalk.utils :as utils]))

;; Data graph builders for the select query result.

(defn get-order [graph entity]
  (la/attr graph entity :order))

(defn set-order [graph entity order]
  (la/add-attr graph entity :order order))

(defn set-table-name [graph entity table-name]
  (la/add-attr graph entity :dbwalk/table-name table-name))

(defn table-name [graph entity]
  (la/attr graph entity :dbwalk/table-name))

(defn set-primary-key-column [graph entity column]
  (la/add-attr graph entity :dbwalk/primary-key-column column))

(defn primary-key-column [graph entity]
  (la/attr graph entity :dbwalk/primary-key-column))

(defn to-insertion-order [graph entities]
  (sort-by #(get-order graph %) entities))

(defn nodes [graph]
  (lg/nodes graph))

(defn- with-insertion-order "Given a seq of items, pairs them with numbers taken in order from [0..length of items-1].
                             This is used to maintain the original order of the items when generating the result tree."
  [items]
  (map-indexed list items))

(defn- add-entity-to-graph [graph config order-and-entity]
  (let [[order entity] order-and-entity
        relation (en/relation entity)
        table    (relation/target-table relation)]
    (-> graph
        (lg/add-nodes entity)
        (set-table-name entity (relation/target-table relation))
        (set-primary-key-column entity (utils/primary-key-column config (relation/to-data-source relation) table))
        (set-order entity order))))

(defn- add-first-level-entities [graph config items]
  (reduce #(add-entity-to-graph %1 config %2) graph (with-insertion-order items)))

(defn add-first-level-entities! "Stores entities without parents."
  [graph-atom config items]
  (swap! graph-atom add-first-level-entities config items))

(defn- add-link-to-graph [graph from to link-data]
  (-> graph
      (lg/add-edges [from to])
      (la/add-attr-to-edges :dbwalk/relation link-data [[from to]])))

(defn- add-child [graph config parent order-and-entity]
  (let [[_ entity] order-and-entity
        relation (en/relation entity)]
    (-> graph
        (add-entity-to-graph config order-and-entity)
        (add-link-to-graph parent entity relation))))


(defn- add-children-to-entity [graph config parent entities]
  (reduce #(add-child %1 config parent %2) graph (with-insertion-order entities)))

(defn- add-children [graph config entity-to-new-children]
  (reduce-kv #(add-children-to-entity %1 config %2 %3) graph entity-to-new-children))

(defn add-children! "Adds new entities as children of their parents.
                     entity-to-new-children is a entity -> seq of children -mapping."
  [graph-atom config entity-to-new-children]
  (swap! graph-atom add-children config entity-to-new-children))

(defn relation-for-link
  [graph from to]
  (la/attr graph from to :dbwalk/relation))

(defn table-for-entity [graph entity]
  (la/attr graph entity :dbwalk/table-name))




(defn find-root-items
  "Returns the nodes without any incoming links."
  [graph]
  (filter #(zero? (lg/in-degree graph %))
          (lg/nodes graph)))

(defn children [graph node]
  (lg/successors @graph node))

(defn predecessors [graph node]
  (lg/predecessors @graph node))

(defn outgoing-edges [graph node]
  (lg/out-edges @graph node))

(defn incoming-edges [graph node]
  (lg/in-edges @graph node))

(defn new-graph []
  (atom (lg/digraph)))

;; Data graph builders for insert. This has to be split because half of the ns takes atoms.

(defn- get-entity-data [graph entity]
  (la/attr graph entity :dbwalk/entity-data))

(defn- set-entity-data [graph entity entity-data]
  (la/add-attr graph entity :dbwalk/entity-data entity-data))

(defn- update-entity-data [graph entity update-f]
  (let [current-data (get-entity-data graph entity)]
    (set-entity-data graph entity (update-f current-data))))

(defn update-entity-data! [graph-atom entity update-f]
  (swap! graph-atom update-entity-data entity update-f))

(defn- add-node-to-graph [graph entity data-source table]
  (-> graph
      (lg/add-nodes entity)
      (la/add-attr entity :dbwalk/table-name table)
      (la/add-attr entity :dbwalk/data-source data-source)
      (set-entity-data entity entity)))                     ;; :dbwalk/entity-data will be updated with foreign keys

(defn add-node! [graph-atom entity data-source table]
  (swap! graph-atom add-node-to-graph entity data-source table))

(defn get-data-for-node [graph-atom entity]
  (la/attrs @graph-atom entity))

(defn remove-node [graph-atom entity]
  (swap! graph-atom lg/remove-nodes entity))

(defn add-link [graph-atom from to data]
  (swap! graph-atom add-link-to-graph from to data))

(defn root-items
  "Finds nodes without incoming edges."
  [graph-atom]
  (find-root-items @graph-atom))

(defn update-key-to-children!
  "After applying an operation to a row, adds the generated primary key (update)
   or nil (delete) as the foreign key to this row's children in the graph."
  [graph-atom entity primary-key]
  (if-let [children (seq (loom.graph/successors @graph-atom entity))]
    (doseq [child children]
      (let [relation (la/attr @graph-atom [entity child] :dbwalk/relation)
            key-name (relation/foreign-key-name relation)]
        (update-entity-data! graph-atom child #(assoc % key-name primary-key))))))

(defn set-operation [graph entity op]
  (swap! graph la/add-attr entity ::operation op))

(defn operation [graph entity]
  (la/attr @graph entity ::operation))

(defn all-nodes [graph]
  (lg/nodes @graph))

(defn table [graph entity]
  (la/attr @graph entity :dbwalk/table-name))


(defn relation-for-edge [graph from to]
  (la/attr @graph from to :dbwalk/relation))

(defn require-update! [graph node]
  (when (= :no-op (operation graph node))
    (set-operation graph node :update)))


