(ns dbwalk.database.sql
  "An implementation for the protocols and multimethods required to use a SQL database as a datasource."
  (:require [clojure.java.jdbc :as jdbc]
            [honeysql.helpers :as hsql]
            [honeysql.core :as sql]
            [clojure.set :refer [union]]
            [dbwalk.relations :as relations]
            [dbwalk.query-parser :as parser]
            [dbwalk.utils :as utils]
            [dbwalk.action.operations :refer [apply-operation!]]))


(defrecord ColumnSpec [table column]
  relations/RelationalData
  (property-name [this]
    column)
  (table [this]
    table))

(defn- unique-target-property-pairs [relations source-items]
  (apply union
    (for [r relations]
      (let [prop (relations/target-property r)]
        (->> source-items
             (map (relations/source-property r))
             (map #(vector prop %))
             (into (hash-set)))))))

(defn- create-where-clause [single-relation-pairs]
  (hsql/where [:in (ffirst single-relation-pairs) (map second single-relation-pairs)]))

(defn- add-merge-clauses-for-relations
  "Generates the HoneySQL query for SELECTing the next level in the query result."
  [query-node property-pairs]
  (let [relation-clauses (->> (group-by first property-pairs)
                              (vals)
                              (map create-where-clause)
                              (mapv :where)
                              (into [:or]))
        query            (parser/query query-node)]
    ;; Just cannot figure out HoneySQL's merge-where
    (if (:where query)
      (hsql/merge-where (dissoc query :where) [:and (:where query) relation-clauses])
      (hsql/merge-where query relation-clauses))))

(defn- add-select-clause-for-properties [query required-properties]
  (apply hsql/select query (seq required-properties)))


(defn single-query [config query-node relations-to-follow nodes-to-advance properties-to-get property-pairs]
  (let [data-source        (relations/data-source-from-relations relations-to-follow)
        data-source-config (relations/select-data-source-from-configuration config data-source)
        query              (-> query-node
                               (add-merge-clauses-for-relations property-pairs)
                               (add-select-clause-for-properties properties-to-get)
                               (#(relations/pre-query-filter config data-source % nodes-to-advance relations-to-follow))
                               (sql/format))]
    (jdbc/query data-source-config query)))


(defmethod relations/advance-query-tree :dbwalk/jdbc-sql [config query-node relations-to-follow nodes-to-advance properties-to-get]
  (let [target-property-pairs (unique-target-property-pairs relations-to-follow nodes-to-advance)]
    (if-let [partition-size (get-in config [:dbwalk/options :dbwalk/partition-size])]
      (->> (partition partition-size partition-size nil target-property-pairs)
           (map (partial single-query config query-node relations-to-follow nodes-to-advance properties-to-get))
           (apply concat))
      (single-query config query-node relations-to-follow nodes-to-advance properties-to-get target-property-pairs))))


(defmethod relations/read-from-table :dbwalk/jdbc-sql [configuration data-source query properties-to-select relations-to-follow]
  (let [query              (apply hsql/select query (seq properties-to-select))
        data-source-config (relations/select-data-source-from-configuration configuration data-source)
        filtered-query     (relations/pre-query-filter configuration data-source query [] relations-to-follow)]
    (jdbc/query data-source-config (sql/format filtered-query))))



;; Action graph operations

(defn- extract-params [configuration data-source-description table entity]
  (let [db-spec            (get-in configuration [:dbwalk/data-sources (:dbwalk/data-source-id data-source-description)])
        primary-key-column (utils/primary-key-column configuration data-source-description table)
        where-clause       [(str (name primary-key-column) " = ?") (get entity primary-key-column)]]
    {:db-spec      db-spec
     :where-clause where-clause}))

;; This multimethod exists to avoid manually dispatching via cond.
(defmulti apply-to-jdbc-database! (fn [configuration data-source-description operation table entity]
                                    operation))

(defmethod apply-to-jdbc-database! :insert [configuration data-source-description operation table entity]
  (let [db-spec (get-in configuration [:dbwalk/data-sources (:dbwalk/data-source-id data-source-description)])]
    (first (jdbc/insert! db-spec table entity))))

(defmethod apply-to-jdbc-database! :delete [configuration data-source-description operation table entity]
  (let [{:keys [db-spec where-clause]} (extract-params configuration data-source-description table entity)]
    (jdbc/delete! db-spec table where-clause)))

(defmethod apply-to-jdbc-database! :update [configuration data-source-description operation table entity]
  (let [{:keys [db-spec where-clause]} (extract-params configuration data-source-description table entity)]
    (first (jdbc/update! db-spec table entity where-clause))))

(defmethod apply-to-jdbc-database! :no-op [configuration data-source-description operation table entity]
  entity)

(defmethod apply-operation! :dbwalk/jdbc-sql [configuration data-source-description operation table entity]
  (apply-to-jdbc-database! configuration data-source-description operation table entity))
