(ns dbwalk.api.subgraph
  "REPL utility for creating QueryPaths. A Builder is an atom containing the database Configuration and the state of the QueryTree being worked on.
   A QueryPath is a directed path which can contain each table at most once."
  (:require [dbwalk.api.tree-map :as tree]
            [dbwalk.relations :as relations]
            [dbwalk.schemas :refer :all]
            [plumbing.core :as pl]
            [schema.core :as s]
            [dbwalk.schemas :refer :all]
            [dbwalk.utils :refer [default-datasource]]))

(def Builder {:dbwalk/data-source   {:dbwalk/data-source-id   s/Keyword
                                     :dbwalk/data-source-type s/Keyword}
              :dbwalk/configuration Configuration
              :dbwalk/query-path    QueryPath})

(s/defn builder-from-config :- (s/atom Builder)
  "Returns a Builder atom initialized from the config. Requires that the config contains only one db."
  [db-config :- Configuration]
  (let [datasources (:dbwalk/data-sources db-config)
        datasource  (default-datasource (first (keys datasources)))]
    (atom {:dbwalk/data-source   datasource
           :dbwalk/configuration db-config
           :dbwalk/query-path    (tree/empty-tree)})))


(defn- all-relations [builder]
  (get-in builder [:dbwalk/configuration :dbwalk/relations (:dbwalk/data-source builder)]))

(defn- all-tables [builder]
  (->> (all-relations builder)
       (map relations/source-table)
       (apply hash-set)))

(defn- all-relations-as-sets [builder]
  (->> (all-relations builder)
       (map #(hash-set (relations/source-table %) (relations/target-table %)))
       (apply hash-set)))


(defn- set-of-targets [xs]
  (apply hash-set (map relations/target-table xs)))

(s/defn possible-links :- {s/Keyword #{s/Keyword}}
  "Lists all possible new links for extending the QueryPath."
  [builder-atom :- (s/atom Builder)]
  (let [builder (deref builder-atom)]
    (->> (all-relations builder)
         (filter #(tree/allowed-link? (:dbwalk/query-path builder) (relations/source-table %) (relations/target-table %)))
         (group-by relations/source-table)
         (pl/map-vals set-of-targets))))

(s/defn targets :- #{s/Keyword}
  "Lists tables the QueryPath can extend to starting from the given table."
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword]
  (let [builder (deref builder-atom)]
    (->> (all-relations builder)
         (filter #(= from (relations/source-table %)))
         (filter #(tree/allowed-link? (:dbwalk/query-path builder) from (relations/target-table %)))
         (map relations/target-table)
         (apply hash-set))))

(s/defn sources :- #{s/Keyword}
  "Lists tables the QueryPath can be extended from."
  [builder-atom :- (s/atom Builder)]
  (let [builder (deref builder-atom)]
    (if (tree/empty-tree? (:dbwalk/query-path builder))
      (all-tables builder)
      (->> (tree/nodes (:dbwalk/query-path builder))
           (filter #(seq (targets builder-atom %)))
           (apply hash-set)))))

(s/defn add-link
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword
   to :- s/Keyword]
  (let [builder   (deref builder-atom)
        possibles (all-relations-as-sets builder)]
    (if (contains? possibles (hash-set from to))
      (swap! builder-atom update :dbwalk/query-path #(tree/add-link % from to)))))

(s/defn remove-link
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword
   to :- s/Keyword]
  (swap! builder-atom update :dbwalk/query-path #(tree/remove-link % from to)))

(s/defn bfs-from
  "Adds all reachable nodes to the path by using a BFS starting from the given node."
  [builder-atom :- (s/atom Builder)
   from :- s/Keyword]
  (if (contains? (all-tables @builder-atom) from)
    (loop [source-nodes #{from}]
      (doall
        (for [source source-nodes]
          (run! #(add-link builder-atom source %) (targets builder-atom source))))
      (let [next-sources (sources builder-atom)]
        (if (seq next-sources)
          (recur next-sources))))
    (throw (IllegalArgumentException. (str "Table " from " does not exist.")))))


(s/defn query-path :- QueryPath
  [builder-atom :- (s/atom Builder)]
  (-> builder-atom
      (deref)
      (:dbwalk/query-path)))

(s/defn build-path-from :- QueryPath
  "Builds a simple QueryPath using a BFS search starting from source-table."
  [db-config :- Configuration
   source-table :- s/Keyword]
  (let [path-builder (builder-from-config db-config)]
    (bfs-from path-builder source-table)
    (query-path path-builder)))

