(ns dbwalk.api.filters.configuration
  (:require [dbwalk.relations :as r]
            [dbwalk.schemas :refer :all]
            [schema.core :as s]))

(s/defn ^:private single-filter
  "Returns a predicate for filtering Relations that have the parameter (:table/column) as one of its ends.
   Returns false (!) for matching Relations."
  [rel-name :- NamespacedKeyword]
  (fn [r]
    (let [table-name  (keyword (namespace rel-name))
          column-name (keyword (name rel-name))]
      (not
        (or
          (and (= (r/source-table r) table-name)
               (= (r/source-property r) column-name))
          (and (= (r/target-table r) table-name)
               (= (r/target-property r) column-name)))))))


(s/defn filter-relations
  "Removes Relations from a Datasource. Filters are given as (:table/column) and will match either end of the Relation.."
  [configuration :- Configuration
   data-source :- DataSourceDescription
   filters :- [NamespacedKeyword]]
  (let [filter-fns         (map single-filter filters)
        relations          (get-in configuration [:dbwalk/relations data-source])
        filtered-relations (filter (apply every-pred filter-fns) relations)]
    (assoc-in configuration [:dbwalk/relations data-source] filtered-relations)))

