(ns vincit.dbwalk.api.detect
  "Utility to detect DB structure. Returns a Loom graph. Can be used for visualization of the database."
  (:require [vincit.dbwalk.schema-detect :as schema]
            [loom.graph :as lg]
            [loom.attr :as la]
            [loom.alg]))


(defn- add-table "Adds a node to represent a table.
                  The node itself will contain the table's name and
                  its attr :table-info will contain everything else except the relations."
  [graph table-info]
  (let [table-name (:name table-info)]
    (-> graph
        (lg/add-nodes table-name)
        (la/add-attr table-name :table-info (dissoc table-info :dbwalk/relations)))))

(defn- add-tables [graph schema]
  (reduce add-table graph schema))

(defn- add-link "Adds a link to a graph to represent a foreign key.
                 The link's attrs will contain the Relation (at :relation)."
  [graph from to relation]
  (-> graph
      (lg/add-edges [from to])
      (la/add-attr-to-edges :relation relation [[from to]])))

(defn- add-links-from [graph table-info]
  (reduce #(add-link %1 (:name table-info) (:target %2) %2) graph (:dbwalk/relations table-info)))

(defn- add-links [graph schema]
  (reduce add-links-from graph schema))

(defn- schema->graph [schema]
  (-> (lg/digraph)
      (add-tables schema)
      (add-links schema)))

(defn detect
  "Autodetects the structure of supported databases. Returns a Loom graph with table names as nodes."
  (
   [db-spec]
   (detect db-spec :default))
  (
   [db-spec type]
   (schema->graph (schema/detect-schema db-spec))))
