(ns vincit.dbwalk.action.full-map
  "
  WORK IN PROGRESS. Everything under vincit.dbwalk.action is only included in master because they are used for tests.
  Not to be used in production.

  Functions for transforming a nested map in the 'map' output format to a Loom graph which
   can be given to v.d.action.writer. The desired operation (see .operations) is given as metadata.
   See .modify for helper functions.

   The nodes in the graph actually contain the full original subtree from the input map. This causes
   only one node to be created for equivalent subtrees in the entire input tree. The effect is a kind of
   duplicate removal."
  (:require [vincit.dbwalk.relations :as relations]
            [vincit.dbwalk.database.sql]
            [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.action.operations :refer [all-operations]]))

(defn- compare-sources [relation to-namespace]
  (= (relations/source-table relation) to-namespace))

(defn- joins-from-table [configuration data-source-description table]
  (let [relations-from-datasource (get-in configuration [:dbwalk/relations data-source-description])]
    (->> relations-from-datasource
         (filter #(compare-sources % table)))))

(defn- joinable-properties-from-table
  "Names of foreign or primary key columns"
  [config data-source-description table]
  (map relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- property-to-relation [config data-source-description table]
  (group-by relations/created-attribute-name (joins-from-table config data-source-description table)))

(defn- detachable? [value]
  (or (map? value)
      (sequential? value)))

(defn- properties-with-detachable-data [entity joinable-properties]
  (let [joinables (apply hash-set joinable-properties)]
    (->> (keys entity)
         (filter #(contains? joinables %))
         (filter #(detachable? (get entity %))))))

(defn- select-properties-to-detach [configuration data-source-description table data]
  (let [joinable-properties (joinable-properties-from-table configuration data-source-description table)]
    (->> data
         (map #(properties-with-detachable-data % joinable-properties))
         (flatten)
         (apply hash-set))))

(defn- simple-get-op [entity]
  (-> (meta entity)
      (select-keys all-operations)
      (first)))

(defn- select-operation [entity primary-key-column default-operation]
  (let [selected-ops (-> (meta entity)
                         (select-keys all-operations))]
    (when (> (count selected-ops) 1)
      (throw (ex-info "Too many operations" {:entity entity :operations selected-ops})))
    (cond
      (nil? (get entity primary-key-column)) :insert
      (= 0 (count selected-ops)) default-operation
      :else (first (keys selected-ops)))))

(defn- link-child [graph entity child relation operation-for-parent operation-for-child]
  (cond
    (= :delete operation-for-child) (graph/add-link graph entity child relation)
    (= :delete operation-for-parent) (graph/add-link graph child entity relation)
    :else (if (relations/insert-source-first relation)
            (graph/add-link graph entity child relation)
            (graph/add-link graph child entity relation))))


(defn- clear-references-to-rows-to-be-deleted
  "When deleting, any references to the deleted entity must be cleared and updated before deleting
  to satisfy the database foreign key constraints. This is not covered by the basic insert-and-push-primary-key
  functionality, so the graph must be prepared before giving it to the writer component."
  [graph]
  (let [nodes-to-be-deleted (->> (graph/all-nodes graph)
                                 (filter #(= (graph/operation graph %) :delete)))]
    (doseq [node-to-be-deleted nodes-to-be-deleted]
      (let [nodes-to-update (graph/predecessors graph node-to-be-deleted)]
        (doseq [node-to-update nodes-to-update]
          (let [table-of-node-to-update (graph/table graph node-to-update)
                relation                (graph/relation-for-edge graph node-to-update node-to-be-deleted)
                relation-end-to-update  (relations/to-description relation)]

            (when (= (relations/table relation-end-to-update) table-of-node-to-update)
              (do
                (graph/update-entity-data! graph node-to-update #(assoc % (relations/property-name relation-end-to-update) nil))
                (graph/require-update! graph node-to-update)))))))))

(defn- to-seq [x]
  (cond
    (nil? x) []
    (sequential? x) x
    :else (vector x)))

(defn build-data-graph [graph configuration data-source-description default-operation table rows]
  (let [properties         (select-properties-to-detach configuration data-source-description table (to-seq rows))
        relation-map       (property-to-relation configuration data-source-description table)
        primary-key-column (utils/primary-key-column configuration data-source-description table)]
    (doseq [entity (to-seq rows)]
      (let [operation-for-entity (select-operation entity primary-key-column default-operation)]
        (graph/add-node! graph entity data-source-description table)
        (graph/set-operation graph entity operation-for-entity)
        (doseq [prop properties]
          (let [children (to-seq (get entity prop []))
                relation (first (get relation-map prop))]
            (build-data-graph graph configuration (relations/to-data-source relation) default-operation (relations/target-table relation) children)
            (doseq [child children]
              (link-child graph entity child relation operation-for-entity (simple-get-op child)))))

        (graph/update-entity-data! graph entity #(apply dissoc % properties)))) ;; Remove properties that represented data in other tables
    (clear-references-to-rows-to-be-deleted graph)))
