(ns vincit.dbwalk.api.simple
  "Simple API for using dbwalk with only one datasource/database. A more feature-rich APi would be unreasonably complex for most use cases.
   Dbwalk considers the data formats (configuration, query, result graph) to be the actual API so that the user can compose their own API functions.
   See vincit.dbwalk.api.vector for a simple way to generate queries."
  (:require [vincit.dbwalk.utils :as utils]
            [vincit.dbwalk.schema-detect :as detect]
            [vincit.dbwalk.schemas :refer :all]
            [vincit.dbwalk.config :as config]
            [vincit.dbwalk.crawler :as crawler]
            [vincit.dbwalk.output.filtered :as filtered]
            [vincit.dbwalk.input.filtered :as filter-in]
            [vincit.dbwalk.api.subgraph :as subgraph]
            [vincit.dbwalk.action.full-map :as action-map]
            [vincit.dbwalk.action.writer :as writer]
            [schema.core :as s]
            [vincit.dbwalk.graph :as graph]))


(defn simple-query->graph
  "Takes a Configuration and a query without datasource information. Uses the first datasource found in the Configuration.
  Returns the Loom graph, which can then be given to an output formatter."
  [configuration query-without-datasource]
  (let [data-source-name (first (keys (:dbwalk/data-sources configuration)))
        data-source      (utils/default-datasource data-source-name)]
    (s/with-fn-validation
      (->> (crawler/run-query configuration (utils/with-datasource data-source query-without-datasource))
           (deref)))))


(defn- filtered-get
  "Takes a Configuration and a query without datasource information. Uses the first datasource found in the Configuration.
  Uses filtered output formatter."

  [configuration columns query-without-datasource]
  (filtered/graph->filtered-map columns
                                (simple-query->graph configuration query-without-datasource)))

(defn- first-namespace [cols]
  (->> cols
       (first)
       (namespace)
       (keyword)))

(defn get-filtered-with-auto-path
  "Takes a Configuration, requested columns as a vector of namespaced keys (:table/column) and a map of {table name -> HoneySQL query}.
  Automatically generates the QueryPath starting from the table of the first key in the vector. Uses the filtered input and output functions."
  (
   [configuration columns base-queries]
   (let [query-path (subgraph/build-path-from configuration (first-namespace columns))
         datasource (utils/default-datasource (first (keys (:dbwalk/data-sources configuration))))]
     (when query-path
       (let [query (utils/with-datasource datasource (filter-in/query-for-columns query-path columns base-queries))]
         (filtered-get configuration columns query)))))
  (
   [configuration columns]
   (get-filtered-with-auto-path configuration columns {})))


(s/defn get-filtered
  "Wrapper around the 'filtered' input and output formatters.
  Selects the given columns (namespaced to tables as in :user/name).
  Returns the data in the filtered output format."
  (
    [configuration :- Configuration
     columns :- [NamespacedKeyword]
     query-path :- QueryPath]
    (filtered-get configuration columns (filter-in/query-for-columns query-path columns)))
  (
    [configuration :- Configuration
     columns :- [NamespacedKeyword]
     query-path :- QueryPath
     base-queries :- s/Any]
    (filtered-get configuration columns (filter-in/query-for-columns query-path columns base-queries))))


(defn- config-for [db-id db-spec]
  (let [database-description (detect/detect-schema db-spec)
        configuration        (merge (config/parse-config database-description (utils/default-datasource db-id))
                                    {:dbwalk/data-sources {db-id db-spec}})]
    configuration))

(defn db-spec->configuration "Creates a simple Configuration from a db-spec."
  [db-spec]
  (config-for (keyword (gensym "simple-db-")) db-spec))


(s/defn with-spec "Updates the db-spec for a datasource. This should be used for transaction management."
  (
    [configuration :- Configuration
     db-id :- s/Any
     db-spec :- s/Any]
    (assoc-in configuration [:dbwalk/data-sources db-id] db-spec))
  (
    [configuration :- Configuration
     db-spec :- s/Any]
    (with-spec configuration (first (keys (:dbwalk/data-sources configuration))) db-spec)))

(s/defn current-spec "Gets the db-spec for a datasource. This and with-spec should be used for transaction management."
  (
    [configuration :- Configuration
     db-id :- s/Any]
    (get-in configuration [:dbwalk/data-sources db-id]))
  (
    [configuration :- Configuration]
    (current-spec configuration (first (keys (:dbwalk/data-sources configuration))))))


;; Writing

(defn action-graph
  "Creates an input graph for the writer component.
  The format of the data must be identical to the 'map' output format.
  Operations (:update / :insert / :delete / :no-op) can be selected by setting them as
  metadata on the maps representing the rows/entities.
  default-operation will be used for all rows/entities without metadata.
  Writing functionality is currently EXPERIMENTAL, do not use in production."
  [configuration data default-operation]
  (let [first-ds    (first (keys (:dbwalk/relations configuration)))
        {:keys [dbwalk/table-name dbwalk/input-data]} data
        input-graph (graph/new-graph)]
    (action-map/build-data-graph input-graph configuration first-ds default-operation table-name input-data)
    input-graph))


(defn- insert! [configuration data-to-insert]
  (let [{:keys [dbwalk/table-name dbwalk/input-data dbwalk/data-source]} data-to-insert
        input-graph (graph/new-graph)]
    (action-map/build-data-graph input-graph configuration data-source :insert table-name input-data)
    (writer/apply-operations configuration input-graph)))


(defn insert-to-first-datasource!
  "Convenience function for writing tests. Inserts the given data to the first datasource found in configuration.
  The format of the data must be identical to the 'map' output format.
  Will not work if the configuration contains more than one datasource.
  Writing functionality is currently EXPERIMENTAL, do not use in production."
  [configuration data-to-insert]
  (let [first-ds (first (keys (:dbwalk/relations configuration)))]
    (insert! configuration (assoc data-to-insert :dbwalk/data-source first-ds))))
