(ns vincit.dbwalk.graph
  "Encapsulates loom.graph."
  (:require [vincit.dbwalk.relations :as relation]
            [loom.graph :as lg]
            [loom.attr :as la]))

;; Data graph builders for the select query result.

(defn get-order [graph entity]
  (la/attr graph entity :order))

(defn set-order [graph entity order]
  (la/add-attr graph entity :order order))


(defn to-insertion-order [graph entities]
  (sort-by #(get-order graph %) entities))


(defn- with-insertion-order "Given a seq of items, pairs them with numbers taken in order from [0..length of items-1].
                             This is used to maintain the original order of the items when generating the result tree."
  [items]
  (map-indexed list items))

(defn- add-entity-to-graph [graph order-and-entity]
  (let [[order entity] order-and-entity]
    (-> graph
        (lg/add-nodes entity)
        (la/add-attr entity :table (relation/target-table (:join entity)))
        (set-order entity order))))

(defn- add-first-level-entities [graph items]
  (reduce #(add-entity-to-graph %1 %2) graph (with-insertion-order items)))

(defn add-first-level-entities! "Stores entities without parents."
  [graph-atom items]
  (swap! graph-atom add-first-level-entities items))

(defn- add-link-to-graph [graph from to link-data]
  (-> graph
      (lg/add-edges [from to])
      (la/add-attr-to-edges :join link-data [[from to]])))

(defn- add-child [graph parent order-and-entity]
  (let [[_ entity] order-and-entity]
    (-> graph
        (add-entity-to-graph order-and-entity)
        (la/add-attr entity :table (relation/target-table (:join entity)))
        (add-link-to-graph parent entity (:join entity)))))


(defn- add-children-to-entity [graph parent entities]
  (reduce #(add-child %1 parent %2) graph (with-insertion-order entities)))

(defn- add-children [graph entity-to-new-children]
  (reduce-kv #(add-children-to-entity %1 %2 %3) graph entity-to-new-children))

(defn add-children! "Adds new entities as children of their parents.
                     entity-to-new-children is a entity -> seq of children -mapping."
  [graph-atom entity-to-new-children]
  (swap! graph-atom add-children entity-to-new-children))

(defn relation-for-link
  [graph from to]
  (la/attr graph from to :join))

(defn table-for-entity [graph entity]
  (la/attr graph entity :table))




(defn find-root-items
  "Returns the nodes without any incoming links."
  [graph]
  (filter #(zero? (lg/in-degree graph %))
          (lg/nodes graph)))

(defn children [graph node]
  (lg/successors @graph node))

(defn predecessors [graph node]
  (lg/predecessors @graph node))

(defn outgoing-edges [graph node]
  (lg/out-edges @graph node))

(defn incoming-edges [graph node]
  (lg/in-edges @graph node))

(defn new-graph []
  (atom (lg/digraph)))

;; Data graph builders for insert

(defn- get-insert-data [graph entity]
  (la/attr graph entity :insert-data))

(defn- set-insert-data [graph entity insert-data]
  (la/add-attr graph entity :insert-data insert-data))

(defn- update-insert-data [graph entity update-f]
  (let [current-data (get-insert-data graph entity)]
    (set-insert-data graph entity (update-f current-data))))

(defn update-insert-data! [graph-atom entity update-f]
  (swap! graph-atom update-insert-data entity update-f))

(defn- add-node-to-graph [graph entity data-source table]
  (-> graph
      (lg/add-nodes entity)
      (la/add-attr entity :table table)
      (la/add-attr entity :data-source data-source)
      (set-insert-data entity entity)))                     ;; :insert-data will be updated with foreign keys

(defn add-node! [graph-atom entity data-source table]
  (swap! graph-atom add-node-to-graph entity data-source table))

(defn get-data-for-node [graph-atom entity]
  (la/attrs @graph-atom entity))

(defn remove-node [graph-atom entity]
  (swap! graph-atom lg/remove-nodes entity))

(defn add-link [graph-atom from to data]
  (swap! graph-atom add-link-to-graph from to data))

(defn root-items
  "Finds nodes without incoming edges."
  [graph-atom]
  (find-root-items @graph-atom))

(defn update-key-to-children!
  "After applying an operation to a row, adds the generated primary key (update)
   or nil (delete) as the foreign key to this row's children in the graph."
  [graph-atom entity primary-key]
  (if-let [children (seq (loom.graph/successors @graph-atom entity))]
    (doseq [child children]
      (let [join (la/attr @graph-atom [entity child] :join)
            key-name (relation/foreign-key-name join)]
        (update-insert-data! graph-atom child #(assoc % key-name primary-key))))))

(defn set-operation [graph entity op]
  (swap! graph la/add-attr entity ::operation op))

(defn operation [graph entity]
  (la/attr @graph entity ::operation))

(defn all-nodes [graph]
  (lg/nodes @graph))

(defn table [graph entity]
  (la/attr @graph entity :table))


(defn relation-for-edge [graph from to]
  (la/attr @graph from to :join))

(defn require-update! [graph node]
  (when (= :no-op (operation graph node))
    (set-operation graph node :update)))


