(ns vincit.dbwalk.output.map
  "The default output, where everything that was read from the DB is returned as a nested map."
  (:require [vincit.dbwalk.entity-node :as en]
            [vincit.dbwalk.graph :as graph]
            [vincit.dbwalk.relations :as relation]))

(defn- get-relation-name [graph from to]
  (relation/created-attribute-name (graph/relation-for-link graph from to)))

(defn data-with-table-namespace
  (
   [graph node table-name]
   (zipmap (map #(keyword table-name (name %)) (keys node)) (vals node)))
  (
   [graph node]
   (let [table-name (name (graph/table-for-entity graph node))]
     (data-with-table-namespace graph node table-name))))

(defn- hierarchy-dump-graph-from-node
  "Recursively dump subgraph starting from node."
  [graph transform start-node]
  (if-let [children (seq (loom.graph/successors graph start-node))]
    (->> children
         (graph/to-insertion-order graph)
         (map #(hash-map
                (get-relation-name graph start-node %)
                [(hierarchy-dump-graph-from-node graph transform %)]))
         (apply merge-with #(vec (concat %1 %2)))
         (merge (transform graph (en/data start-node))))
    (transform graph (en/data start-node))))

(defn- to-map-with-transform
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :eager query. }"
  [graph transform]
  (en/data-only
    (->> (graph/find-root-items graph)
         (graph/to-insertion-order graph)
         (map (partial hierarchy-dump-graph-from-node graph transform))
         (vec))))

(defn to-map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :eager query. }"
  [graph]
  (to-map-with-transform graph (fn [a b] b)))

(defn to-namespaced-map
  "Returns the graph as a list of root entities, where each entity
  is merged with { tablename -> seq of eagerly fetched items for each :eager query. }"
  [graph]
  (to-map-with-transform graph data-with-table-namespace))
